/**
* js/pianoroll.js
*
* pianoroll basic component
*
*/

'use strict';

var PIXI = require('pixi');
var randomColor = require('randomColor');

function PianoRoll(parentContainer, xInit, yInit, height, linesDown, pixelsPerSecond, width, noteColors, colorsReg, lineColor, lineInterval, offsetMusic, noteHeight, horizontalView){
    var _this = this;
    this.container = new PIXI.DisplayObjectContainer();
    this.container.position.x = xInit;
    this.container.position.y = yInit;
    if (!horizontalView){
    	this.container.width = width;
    }
    parentContainer.addChild(this.container);

    this.linesDown = linesDown;
    this.height = height;
    this.pixelsPerSecond = pixelsPerSecond;
    this.width = width;
    this.noteColors = noteColors;
    this.colorsReg = colorsReg || {};
    this.lineColor = lineColor;
    this.lineInterval = lineInterval;
    this.offsetMusic = offsetMusic || 0;
    this.noteHeight = noteHeight;

    this.addNote = function(note, startTime, duration, velocity, canal){
        var begin = (this.offsetMusic + startTime) * this.pixelsPerSecond / 1000;
        if (horizontalView){
	        var width = duration * this.pixelsPerSecond / 1000;
	        if((begin+width) <  (Math.abs(this.container.x) - this.width)) {
	            // not visible. do nothing
	            return;
	        }
        } else {
        	var height = duration * this.pixelsPerSecond / 1000;
	        if((begin+height) <  (Math.abs(this.container.y) - this.height)) {
	            // not visible. do nothing
	            return;
	        }
        }
        // We draw the rectangle
        var graphics = new PIXI.Graphics();
        var color = this.colorsReg[canal];
        if(typeof(color) === 'undefined') {
            var colorsRegSize = Object.keys(this.colorsReg).length;
            if(colorsRegSize < this.noteColors.length) {
                color = this.colorsReg[canal] = this.noteColors[colorsRegSize];
            }
            else {
                color = this.colorsReg[canal] = parseInt(randomColor({ luminosity: 'light', hue: 'random', format:'hex'}).replace(/^#/, ''), 16);
            }
        }
        graphics.beginFill(color, (velocity / 128));
        if (horizontalView){
	        var y = (128-note) * this.height / 128; // (128-note) because y = 0 is for note = 128 and y = 128 for note = 0
	        graphics.drawRect(0, Math.floor(y - (noteHeight/2) + ((this.height / 128)/2)), width, noteHeight);
	        graphics.endFill();
	        graphics.x = begin;
        } else {
        	var x = (128-note) * this.width / 128; // (128-note) because y = 0 is for note = 128 and y = 128 for note = 0
	        graphics.drawRect(Math.floor(x - (noteHeight/2) + ((this.width / 128)/2)), 0, noteHeight, height);
	        graphics.endFill();
	        graphics.y = begin;
        }
        this.container.addChild(graphics);
    };

    this.addLine = function(ts){
        var graphics = new PIXI.Graphics();
        if (horizontalView){
        	var x = -this.container.x;
        	var y = this.linesDown ? this.height - 20 : 0;
        } else {
        	var x = this.linesDown ? 0 : this.width - 20 ;
        	var y = -this.container.y;
        }
        graphics.beginFill(0xFFFF00);
        graphics.lineStyle(1, this.lineColor);
        graphics.moveTo(x, y);
        if (horizontalView){
        	graphics.lineTo(x, y + 20);
        } else {
        	graphics.lineTo(x + 20, y);
        }
        graphics.endFill();
        this.container.addChild(graphics);
        // Add text
        //var totalSec = lineNb * this.lineInterval / 1000;
        var hours = ts.getHours();
        var minutes =ts.getMinutes();
        var seconds = ts.getSeconds();
        var timeStr = (hours < 10 ? '0' + hours : hours) + ':' + (minutes < 10 ? '0' + minutes : minutes) + ':' + (seconds  < 10 ? '0' + seconds : seconds);

        var fontObj = { font: '10pt Arial', fill: '#444444' };
        var t = new PIXI.Text(timeStr, fontObj);
        if (horizontalView){
	        t.x = x + 2;
	        t.y = this.linesDown ? this.height - 15 : 2;
        } else {
	        t.x = this.linesDown ? 2 : this.width - 55;
	        t.y = y + 2;
        }
        this.container.addChild(t);
    };

    this.moveTo = function(diffTime){
    	if (horizontalView){
    		this.container.x = Math.floor(diffTime*this.pixelsPerSecond);
    	} else {
    		this.container.y = Math.floor(diffTime*this.pixelsPerSecond);
    	}
    };

    this.removePassedObjets = function(){
        var nbChilds = _this.container.children.length;
        var i = 0, childIsNowDisplayed = false, childrenToRemove = [];
        while(i<nbChilds && !childIsNowDisplayed){
            var child = _this.container.children[i++];
            if(typeof(child) == 'undefined') {
                continue;
            }
            if (horizontalView){
	            if((child.x + child.width) < (Math.abs(_this.container.x) - _this.width)){
	                childrenToRemove.push(child);
	            }
	            else {
	                childIsNowDisplayed = true;
	            }
            } else {
            	if((child.y + child.height) < (Math.abs(_this.container.y) - _this.height)){
	                childrenToRemove.push(child);
	            }
	            else {
	                childIsNowDisplayed = true;
	            }
            }
        }
        childrenToRemove.forEach(function(child) {
            _this.container.removeChild(child);
        });
    };

    // remove notes each scene width
    //var removeInterval = window.setInterval(this.removePassedObjets, 1000 * sceneWidth / this.pixelsPerSecond );
    if (horizontalView){
    	window.setInterval(this.removePassedObjets, 1000 * this.width / this.pixelsPerSecond );
    } else {
    	window.setInterval(this.removePassedObjets, 1000 * this.height / this.pixelsPerSecond );
    }

}

module.exports = PianoRoll;
