/**
* js/annotstimeline
*
* annotstimeline basic component
*
*/

'use strict';

var PIXI = require('pixi');
var Utils = require('./utils.js');
var _ = require('lodash');

var defaultOptions = {
    logger: undefined,
    intervalWidth: 10,
    intervalHeight: 5,
    maxCellHeight: 200,
    radius: 300
};


function AnnotsTimeLine(options){
    var _this = this;
    var opts = _(options).defaults(defaultOptions).value();

    this.container = new PIXI.DisplayObjectContainer();
    this.container.x = opts.xInit;
    this.container.y = opts.yInit;
    this.container.width = opts.width;
    this.container.height = opts.height;

    this.timeBegin = opts.timeBegin;
    this.timeEnd = opts.timeEnd;
    this.duration = (this.timeEnd - this.timeBegin)/1000;
    this.width = opts.width;
    this.height = opts.height;
    this.intervalHeight = opts.intervalHeight;
    this.intervalWidth = opts.intervalWidth;
    this.maxCellHeight = opts.maxCellHeight;
    this.annotCategories = opts.annotCategories;

    this.circleX = opts.circleX || (this.width/2);
    this.circleY = opts.circleY || (this.height/2);
    this.radius = opts.radius;
    this.perimeter = 2*Math.PI* this.radius;
    this.intervalDuration = (this.intervalWidth * this.duration / this.perimeter);

    var currentTime = this.timeBegin;
    var totalIndex = Math.floor(this.perimeter/this.intervalWidth);

    this.cells = []
    for (var i=0; i<(this.perimeter/this.intervalWidth) ; i++){
    	this.cells[i] = [];
    	this.cells[i].i = i;
    	this.cells[i].totalAnnots = 0;
    	this.cells[i].categories = {};

    	for (var category in this.annotCategories[0].colors){
    		this.cells[i].categories[category] = {
				"count": 0,
				"color": this.annotCategories[0].colors[category]
    		};
    	}
    }

    var ws = opts.ws;
    var stageView = opts.stageView;

    //draw the base - circle and line to locate the scene
    var graphics = new PIXI.Graphics();
    graphics.lineStyle(2, 0x646464)
    	.drawCircle(this.circleX, this.circleY, this.radius - 3)
    	.lineStyle(1, 0xD7D7D7)
    	.drawCircle(this.circleX, this.circleY, this.radius*2/3)
    	.drawCircle(this.circleX, this.circleY, this.radius/3)
    	.lineStyle(1, 0x646464)
    	.moveTo(this.circleX, this.circleY - (this.radius/3)/2)
    	.lineTo(this.circleX, this.circleY - this.radius - this.maxCellHeight - 10)
    	.endFill()
    this.container.addChild(graphics);

    //set time text
    //TODO : move this to annotsvizview
    var currentTimeText = new PIXI.Text("-- : -- : --", { font: '18pt Gothic Standard', fill: '#646464' });
    currentTimeText.x = this.circleX - currentTimeText.width/2;
    currentTimeText.y = this.circleY - currentTimeText.height/2;
    this.container.addChild(currentTimeText);

    stageView.registerComponent(this);

    //Add Annotation to the TimeLine
    this.addAnnot = function(data){
    	if (typeof(this.annotCategories[0].colors[data.content.category.code]) !== 'undefined'){
    		var annotCode = data.content.category.code;
    	} else {
    		var annotCode = this.annotCategories[0].order[this.annotCategories[0].order.length -1];
    	}
    	var annotTime = Date.parse(data.ts);

    	if (this.timeEnd > Date.parse(data.ts)){
	    	var i = Math.floor((Date.parse(data.ts)-this.timeBegin)/(1000*this.intervalDuration));

			this.cells[i].categories[annotCode].count += 1;
			this.cells[i].totalAnnots +=1;
			this.redrawCell(this.cells[i], i);
    	}
    };

    this.initGraphics = function(cell){
    	cell.graphics = new PIXI.Graphics();
    	cell.graphics.position.x = this.circleX + this.radius * Math.sin(cell.i*(360/totalIndex)*(Math.PI/180));
    	cell.graphics.position.y = this.circleY - this.radius * Math.cos(cell.i*(360/totalIndex)*(Math.PI/180));
    	cell.graphics.rotation = (cell.i)*(360/totalIndex)*(Math.PI/180) + (360/(totalIndex*2))*(Math.PI/180);
    	this.container.addChild(cell.graphics);
    }

    this.initTimeTexts = function() {
	    var tBeg = new PIXI.Text(Utils.formatTime(this.timeBegin), { font: '12pt Gothic Standard', fill: '#646464' });
	    tBeg.x = this.circleX + 15;
	    tBeg.y = this.circleY - this.radius - this.maxCellHeight - 10;
	    this.container.addChild(tBeg);

	    var tEnd = new PIXI.Text(Utils.formatTime(this.timeEnd), { font: '12pt Gothic Standard', fill: '#646464' });
	    tEnd.x = this.circleX - 15 - tEnd.width;
	    tEnd.y = this.circleY - this.radius - this.maxCellHeight - 10;
	    this.container.addChild(tEnd);

	    var t15 = new PIXI.Text(Utils.formatTime(((this.timeEnd - this.timeBegin)/4) + this.timeBegin), { font: '12pt Gothic Standard', fill: '#646464' });
	    t15.x = this.circleX + this.radius + this.maxCellHeight + 10 ;
	    t15.y = this.circleY - t15.height;
	    t15.rotation = Math.PI /2;
	    this.container.addChild(t15);

	    var t30 = new PIXI.Text(Utils.formatTime(((this.timeEnd - this.timeBegin)/2) + this.timeBegin), { font: '12pt Gothic Standard', fill: '#646464' });
	    t30.x = this.circleX - t30.width/2;
	    t30.y = this.circleY + this.radius + this.maxCellHeight - 2;
	    this.container.addChild(t30);

	    var t45 = new PIXI.Text(Utils.formatTime(((this.timeEnd - this.timeBegin)*3/4) + this.timeBegin), { font: '12pt Gothic Standard', fill: '#646464' });
	    t45.x = this.circleX - this.radius - this.maxCellHeight - 10 ;
	    t45.y = this.circleY + t15.height;
	    t45.rotation = -Math.PI/2;
	    this.container.addChild(t45);
    }

    //Draw the cellule
    this.redrawCell = function(cell){

    	if (typeof(cell.graphics) === 'undefined'){
    		this.initGraphics(cell);
    	} else {
    		cell.graphics.clear();
    	}

    	var y = 0;

    	//Check if total height is higher than Max Cell Height
    	if ((cell.totalAnnots*this.intervalHeight) > this.maxCellHeight){
    		var heightStep = this.maxCellHeight/cell.totalAnnots;
    	} else {
    		var heightStep = this.intervalHeight;
    	}

    	//Draw the rect depending on the height step calculated
    	for (var i=0; i< this.annotCategories[0].order.length; i++){
    		var currentCode = this.annotCategories[0].order[i];
			cell.graphics.beginFill(cell.categories[currentCode].color.replace("#", "0x"))
    			.drawRect(0, y, this.intervalWidth-1, -cell.categories[currentCode].count * heightStep)
    			.endFill();
    		y -= cell.categories[currentCode].count*heightStep;
    	}
    }

    this.init = function() {
    	ws.message(function(data) {
            _this.addAnnot(data);
        });

    	this.initTimeTexts();
    };

    this.updateTime = function(){
    	currentTime += 1000;

        var nbSec = currentTime / 1000;
        var hours = Math.floor( nbSec / 3600 ) % 24;
        var minutes = Math.floor( nbSec / 60 ) % 60;
        var seconds = Math.floor(nbSec % 60);
        var timeStr = (hours < 10 ? '0' + hours : hours) + ':' + (minutes < 10 ? '0' + minutes : minutes) + ':' + (seconds  < 10 ? '0' + seconds : seconds);

        currentTimeText.setText(timeStr);
    };

    var refreshTimeInterval;

    this.start = function() {
    	refreshTimeInterval = setInterval(function() {_this.updateTime();}, 1000);
    };

    this.refresh = function() {

    };

    this.stop = function(){
    	console.log(this.cells);
    };

    return this;
}

module.exports = {
	AnnotsTimeLine: AnnotsTimeLine
};
