/**
 * scripts/main.js
 *
 * This is the starting point for your application.
 * Take a look at http://browserify.org/ for more info
 */

 /* global window: false */
 /* global document: false */
 /* global WebSocket: false */
 /* global MozWebSocket: false */

 /* global eventCode: false */
'use strict';


var PIXI = require('pixi');
var _ = require('lodash');


var NTP_EPOCH_DELTA = 2208988800; //c.f. RFC 868
// Config vars
var logger = false;
var sceneWidth = 1920;
var prHeight1 = 435;
var prHeight2 = 645;
var sceneHeight = prHeight1 + prHeight2;
var sceneBgColor = 0xFFFFFF;
var lineColor = 0x444444;
var pixelsPerSecond1 = Math.floor(sceneWidth / 10); // nb of pixels per second
var manualFramerate = pixelsPerSecond1 / 4;
var pixelsPerSecond2 = Math.floor(sceneWidth / 60); // nb of pixels per second
var lineInterval = 5000; // means line every 5 seconds
//var noteHeight = 110;
var noteColors = [0xB90000, 0x4BDD71, 0xAF931E, 0x1C28BA, 0x536991];
var colorsReg = {};
// Timecode method
var offsetMusic = false;


//create an new instance of a pixi stage
var stage = new PIXI.Stage(sceneBgColor);

//create a renderer instance.
var renderer = PIXI.autoDetectRenderer(sceneWidth, sceneHeight);

//add the renderer view element to the DOM
document.getElementById('canvasContainer').appendChild(renderer.view);

var uberContainer = new PIXI.DisplayObjectContainer();
uberContainer.position.x = Math.floor(sceneWidth*9/10);
uberContainer.position.y = 0;
stage.addChild(uberContainer);

var PianoRoll = require('./pianoroll.js');

// Init containers
var containerList = [];

var pianorollOptions = {
    parentContainer: uberContainer,
    xInit: 0,
    width: sceneWidth,
    noteColors: noteColors,
    colorsReg: colorsReg,
    lineColor: lineColor,
    lineInterval: lineInterval,
    offsetMusic: offsetMusic
};


containerList.push(new PianoRoll(_.extend(_.clone(pianorollOptions), {
    yInit: 0,
    height: prHeight1,
    linesDown: true,
    pixelsPerSecond: pixelsPerSecond1,
    noteHeight: prHeight1 / 128
})));
containerList.push(new PianoRoll(_.extend(_.clone(pianorollOptions), {
    yInit: prHeight1,
    height: prHeight2,
    linesDown: false,
    pixelsPerSecond: pixelsPerSecond2,
    noteHeight: prHeight2 / 128
})));


// Line between two containers
var graphics = new PIXI.Graphics();
graphics.beginFill(0xFFFF00);
graphics.lineStyle(1, lineColor);
graphics.moveTo(0, prHeight1);
graphics.lineTo(sceneWidth, prHeight1);
graphics.endFill();
stage.addChild(graphics);


function addNotes(data){
    var note = data.content[3];
    var velocity = data.content[4];
    var ts = (data.content[0] - NTP_EPOCH_DELTA)*1000;
    var channel = data.content[2];
    var sessionTs = data.content[1];

    containerList.forEach(function(c) {
        c.addNote(note, ts, sessionTs, velocity, channel, 0);
    });
}



// Socket management
var sock = null;
var ellog = null;
function log(m) {
    if(logger){
        ellog.innerHTML += m + '\n';
        ellog.scrollTop = ellog.scrollHeight;
    }
}
window.onload = function(){

    if(logger){
        ellog = document.getElementById('log');
    }
    else{
        document.body.removeChild(document.getElementById('log'));
    }

    var wsuri;
    if (window.location.protocol === 'file:') {
        wsuri = 'ws://127.0.0.1:8090/broadcast';
    } else {
        wsuri = 'ws://' + window.location.hostname + ':8090/broadcast';
    }
    wsuri = wsuri + '?channel=PIANOROLL&event_code='+eventCode;

    if ('WebSocket' in window) {
        sock = new WebSocket(wsuri);
    } else if ('MozWebSocket' in window) {
        sock = new MozWebSocket(wsuri);
    } else {
        log('Browser does not support WebSocket!');
        window.location = 'http://autobahn.ws/unsupportedbrowser';
    }

    if (sock) {
        sock.onopen = function(){
            if(logger){
                log('Connected to ' + wsuri);
            }
        };

        sock.onclose = function(e) {
            if(logger){
                log('Connection closed (wasClean = ' + e.wasClean + ', code = ' + e.code + ', reason = \'' + e.reason + '\')');
            }
            sock = null;
        };

        sock.onmessage = function(e) {
            var dataJson = JSON.parse(e.data);
            var dataDate = new Date((dataJson.content[0]-NTP_EPOCH_DELTA)*1000);
            if(logger){
                log('Got message: ' + e.data + ' - ' + dataDate.toISOString());
            }
            addNotes(dataJson);
        };
    }
};

function refreshStage() {
    containerList.forEach(function(c) {
        c.move();
    });
    renderer.render(stage);
}


// Init page and intervals
var refreshInterval;

function start() {
    refreshInterval = window.setInterval(refreshStage, 1000/manualFramerate);
    containerList.forEach(function(c) {
        c.start();
    });
}

function stop() {
    window.clearInterval(refreshInterval);
    containerList.forEach(function(c) {
        c.stop();
    });
}

// Little interval to show time
var nbSec = 0;
var mySpan = document.getElementById('myspan');
function updateTime(){
    nbSec++;
    var hours = parseInt( nbSec / 3600 ) % 24;
    var minutes = parseInt( nbSec / 60 ) % 60;
    var seconds = nbSec % 60;
    var timeStr = (hours < 10 ? '0' + hours : hours) + ':' + (minutes < 10 ? '0' + minutes : minutes) + ':' + (seconds  < 10 ? '0' + seconds : seconds);
    mySpan.innerHTML = timeStr;
}
window.setInterval(updateTime, 1000);

start();

module.exports = {
    start: start,
    stop: stop,
};
