#
# See LICENCE for detail
# Copyright (c) 2014 IRI
#

import json
import urllib

from flask.ext.admin import Admin
from flask.ext.admin.contrib.sqla import ModelView
from flask.ext.admin.contrib.sqla.form import AdminModelConverter
from flask.ext.admin.contrib.sqla.filters import BooleanEqualFilter
from flask.ext.admin.model.form import InlineFormAdmin, converts

from jinja2 import Markup

from wtforms import fields
from wtforms.ext.dateutil import fields as dateutil_fields

import database
import models
import config
from webapp import app

admin = Admin(app)


def build_ldt_url(view, context, model, name):
    # `view` is current administrative view
    # `context` is instance of jinja2.runtime.Context
    # `model` is model instance
    # `name` is property name
    if not model.project_id \
        or not model.categories_json \
        or not isinstance(model.categories_json, dict) \
        or not getattr(config,'LDT_PLATFORM_BASE_URL',None):
        return ''

    output = []

    for c in model.categories_json.get("categories",[]):
        if "subcategories" in c and len(c["subcategories"])>0:
            for sc in c["subcategories"]:
                output.append({"name": sc["code"],"keywords": [ sc["label"] ], "color" : sc["color"] })
        else:
            output.append({"name": c["code"],"keywords": [ c["label"] ], "color" : c["color"] })

    base_url = getattr(config,'LDT_PLATFORM_BASE_URL','') + "/ldt/embediframe/?"
    url_query_params = {'project_id': model.project_id, 'polemic': 'all', 'polemics_list': json.dumps(output, separators=(',',':'))}

    url = base_url + urllib.urlencode(url_query_params)

    return Markup("<a href=\"%s\" target=\"_blank\">%s</a>" % (url,url[:80]+"..."))


class EventView(ModelView):
    column_searchable_list = ('code', 'label')
    column_list = ('code', 'label', 'start_date', 'active')
    column_filters = ('code', 'label', 'start_date', 'active')
    form_create_rules = ('code', 'label', 'description', 'start_date')
    inline_models = (models.EventSession,)
    #form_edit_rules = ('label', 'description', 'start_date', 'active', 'code')

    def __init__(self, session, **kwargs):
        super(EventView, self).__init__(models.Event, session, **kwargs)


class DictToJSONField(fields.TextAreaField):
    def process_data(self, value):
        if value is None:
            value = {}

        self.data = json.dumps(value, indent=2)

    def process_formdata(self, valuelist):
        if valuelist:
            self.data = json.loads(valuelist[0])
        else:
            self.data = '{}'

class JsonAdminConverter(AdminModelConverter):
    @converts('JSON')
    def conv_JSON(self, field_args, **extra):
        return DictToJSONField(**field_args)

class EventSessionView(ModelView):

    column_default_sort = ( models.Event.code, True)
    column_sortable_list = (('event', models.Event.code),)
    column_auto_select_related = True
    column_list = ('event', 'project_id', 'order', 'categories_json', 'url')
    column_filters = ('event',)
    column_formatters = {'url': build_ldt_url, 'categories_json': lambda v, c, m, p: json.dumps(m.categories_json)[:80] if m.categories_json else ''}
    form_args = {
        "start_ts":{ "display_format": '%Y-%m-%dT%H:%M:%S.%f%z'},
    }
    form_overrides = { "start_ts": dateutil_fields.DateTimeField, }
    form_widget_args = {
        'categories_json': {
            'rows': 10,
            'class': 'input-xlarge'
        },
        'start_ts': {
            'class': 'input-xlarge'
        },
        'project_id': {
            'class': 'input-xlarge'
        }
    }

    model_form_converter=JsonAdminConverter

    def __init__(self, session, **kwargs):
        super(EventSessionView, self).__init__(models.EventSession, session, **kwargs)



admin.add_view(EventView(database.db_session))
admin.add_view(EventSessionView(database.db_session))
