/**
* js/annotstimeline
*
* annotstimeline basic component
*
*/

'use strict';

var PIXI = require('pixi');
var Utils = require('./utils.js');
var _ = require('lodash');

var defaultOptions = {
    logger: undefined,
    intervalWidth: 10,
    intervalHeight: 5,
    maxCellHeight: 200,
    radius: 300,
    serverUrl: 'http://127.0.0.1:8080',
    channel: 'ANNOT',
    maxPages: 1000,
    showClockGraphics: true,
    archive: false
};


function AnnotsTimeLine(options){
    var _this = this;
    var opts = _(options).defaults(defaultOptions).value();

    this.container = new PIXI.DisplayObjectContainer();
    this.container.x = opts.xInit;
    this.container.y = opts.yInit;
    this.container.width = opts.width;
    this.container.height = opts.height;

    this.timeBegin = opts.timeBegin;
    this.timeEnd = opts.timeEnd;
    this.duration = (this.timeEnd - this.timeBegin)/1000;
    this.width = opts.width;
    this.height = opts.height;
    this.intervalHeight = opts.intervalHeight;
    this.intervalWidth = opts.intervalWidth;
    this.maxCellHeight = opts.maxCellHeight;
    this.annotCategories = opts.annotCategories;
	this.startTs = options.startTs || Date.now();
    this.showClockGraphics = opts.showClockGraphics;
    this.archive = opts.archive;
    
    this.circleX = opts.circleX || (this.width/2);
    this.circleY = opts.circleY || (this.height/2);
    this.radius = opts.radius;
    var perimeter = 2*Math.PI* this.radius;
    this.intervalDuration = (this.intervalWidth * this.duration / perimeter);
    
    var channel = opts.channel;
    var eventCode = opts.eventCode;
    var serverUrl = opts.serverUrl;
    var maxPages = opts.maxPages;
    
    var totalIndex = Math.floor( perimeter/this.intervalWidth);

    this.cells = []
    for (var i=0; i<(perimeter/this.intervalWidth) ; i++){
    	this.cells[i] = [];
    	this.cells[i].i = i;
    	this.cells[i].totalAnnots = 0;
    	this.cells[i].categories = {};
    }

    var ws = opts.ws;
    var stageView = opts.stageView;

    //draw the base - circle and line to locate the scene
    var graphics = new PIXI.Graphics();
    graphics.lineStyle(2, 0x646464)
    	.drawCircle(this.circleX, this.circleY, this.radius - 3)
    	.endFill()
    this.container.addChild(graphics);

    stageView.registerComponent(this);

	var loadArchives = function() {
        //start timeBegin end startTime
        //query -> need channel + eventCode
        //iterate over data fill cells
        var startTs = _this.timeBegin;
        var endTs = _this.startTs;

        var url = serverUrl + '/p/api/v1/annotation';
        var filters = [
            { name: 'ts', op: '>', val: new Date(startTs).toISOString()}, //start
            { name: 'ts', op: '<=', val: new Date(endTs).toISOString()}, //end
            { name: 'channel', op: '==', val: channel}, //channel
            { name: 'event_code', op: '==', val: eventCode} //eventcode
        ];

        url = url + '?q=' + JSON.stringify({filters:filters});

        var totalPage = 1;
        var currentPage = 1;

        var processResFunction = function(res) {

            if(res) {
                var data = res.target.json;
                /*jshint -W069 */
                totalPage = Math.min(maxPages,parseInt(data['total_pages']));
                data.objects.forEach(function(annotation) {
                    _this.addAnnot(annotation);
                });
            }
            if(currentPage <= totalPage) {
                var jsonLoader = new PIXI.JsonLoader(url+'&page='+currentPage, true);
                jsonLoader.on('loaded', processResFunction);
                jsonLoader.load();
                currentPage++;
            }
        };
        processResFunction();

    };

    //Add Annotation to the TimeLine
    this.addAnnot = function(data){

    	var ts = Date.parse(data.ts);
    	var colorsDef;
    	_(this.annotCategories).eachRight(function(cdef) {
            if(cdef.ts < ts) {
                colorsDef = cdef;
                return false;
            }
        });

    	if (this.timeEnd > ts){
	    	var i = Math.floor((ts - this.timeBegin)/(1000*this.intervalDuration));

	    	if (typeof(this.cells[i].graphics) === 'undefined'){
	    		this.initCell(this.cells[i], colorsDef);
	    	}

	    	if (typeof(colorsDef.colors[data.content.category.code]) !== 'undefined'){
	    		var annotCode = data.content.category.code;
	    	} else {
	    		var annotCode = 'default';
	    	}

			this.cells[i].categories[annotCode].count += 1;
			this.cells[i].totalAnnots +=1;
			this.redrawCell(this.cells[i], colorsDef);
    	}
    };

    this.initClockGraphics = function() {
	    var tBeg = new PIXI.Text(Utils.formatTime(this.timeBegin), { font: '12pt Gothic Standard', fill: '#646464' });
	    tBeg.x = this.circleX + 15;
	    tBeg.y = this.circleY - this.radius - this.maxCellHeight - 10;
	    this.container.addChild(tBeg);

	    var tEnd = new PIXI.Text(Utils.formatTime(this.timeEnd), { font: '12pt Gothic Standard', fill: '#646464' });
	    tEnd.x = this.circleX - 15 - tEnd.width;
	    tEnd.y = this.circleY - this.radius - this.maxCellHeight - 10;
	    this.container.addChild(tEnd);

	    var t15 = new PIXI.Text(Utils.formatTime(((this.timeEnd - this.timeBegin)/4) + this.timeBegin), { font: '12pt Gothic Standard', fill: '#646464' });
	    t15.x = this.circleX + this.radius + this.maxCellHeight + 10 ;
	    t15.y = this.circleY - t15.height;
	    t15.rotation = Math.PI /2;
	    this.container.addChild(t15);

	    var t30 = new PIXI.Text(Utils.formatTime(((this.timeEnd - this.timeBegin)/2) + this.timeBegin), { font: '12pt Gothic Standard', fill: '#646464' });
	    t30.x = this.circleX - t30.width/2;
	    t30.y = this.circleY + this.radius + this.maxCellHeight - 2;
	    this.container.addChild(t30);

	    var t45 = new PIXI.Text(Utils.formatTime(((this.timeEnd - this.timeBegin)*3/4) + this.timeBegin), { font: '12pt Gothic Standard', fill: '#646464' });
	    t45.x = this.circleX - this.radius - this.maxCellHeight - 10 ;
	    t45.y = this.circleY + t15.height;
	    t45.rotation = -Math.PI/2;
	    this.container.addChild(t45);
	    
	    var lineV = new PIXI.Graphics();
	    lineV.lineStyle(1, 0x646464)
	    	.moveTo(this.circleX, this.circleY - (this.radius/3)/2)
	    	.lineTo(this.circleX, this.circleY - this.radius - this.maxCellHeight - 10)
    	.endFill();
	    this.container.addChild(lineV);
    }

    //Draw the cellule
    this.redrawCell = function(cell, colorsDef){
    	var y = 0;

        //Check if total height is higher than Max Cell Height
        var heightStep;
        if ((cell.totalAnnots*this.intervalHeight) > this.maxCellHeight){
            heightStep = this.maxCellHeight/cell.totalAnnots;
        } else {
            heightStep = this.intervalHeight;
        }

    	//Draw the rect depending on the height step calculated
    	for (var i=0; i< colorsDef.order.length; i++){
    		var currentCode = colorsDef.order[i];
			cell.graphics.beginFill(cell.categories[currentCode].color.replace("#", "0x"))
    			.drawRect(0, y, this.intervalWidth-1, -cell.categories[currentCode].count * heightStep)
    			.endFill();
    		y -= cell.categories[currentCode].count*heightStep;
    	}
    }

    this.initCell = function(cell, colorsDef){
    	cell.graphics = new PIXI.Graphics();
    	cell.graphics.position.x = this.circleX + this.radius * Math.sin(cell.i*(360/totalIndex)*(Math.PI/180));
    	cell.graphics.position.y = this.circleY - this.radius * Math.cos(cell.i*(360/totalIndex)*(Math.PI/180));
    	cell.graphics.rotation = (cell.i)*(360/totalIndex)*(Math.PI/180) + (360/(totalIndex*2))*(Math.PI/180);
    	this.container.addChild(cell.graphics);

    	for (var category in colorsDef.colors){
    		cell.categories[category] = {
				"count": 0,
				"color": colorsDef.colors[category]
    		};
    	}
    	if (typeof(cell.categories['default']) === 'undefined'){
			cell.categories['default'] = {
				"count": 0,
				"color": colorsDef.defaultColor
			}
    	}
    }

    this.init = function() {
    	if (!this.archive){
	        ws.message(function(data) {
	            _this.addAnnot(data);
	        });
    	}

    	if (this.showClockGraphics){this.initClockGraphics();}
    };


    this.start = function() {
    	this.startTs = Date.now();
        loadArchives();
    };

    this.refresh = function() {

    };

    this.stop = function(){
    };

    return this;
}

module.exports = {
	AnnotsTimeLine: AnnotsTimeLine
};
