/**
* scripts/stageview.js
*
* This is the starting point for your application.
* Take a look at http://browserify.org/ for more info
*/

/* global document: false */

'use strict';


var PIXI = require('pixi');
var _ = require('lodash');

var defaultConfig = {
    externalRefresh: false,
    logger: undefined,
    sceneWidth: 1024,
    sceneHeight: 768,
    framerate: 25,
    sceneBgColor: 0xFFFFFF,
    canvasContainer: 'canvasContainer',
    timeContainer: 'timeStarted'
};

function StageView(options) {

    var _this = this;
    var opts = _(options).defaults(defaultConfig).value();

    var externalRefresh = opts.externalRefresh;

    this.logger = opts.logger;
    this.framerate = opts.framerate;
    var sceneBgColor = opts.sceneBgColor;
    var sceneWidth = opts.sceneWidth;
    var sceneHeight = opts.sceneHeight;
    var canvasContainer = opts.canvasContainer;
    var timeContainer = opts.timeContainer;

    //create an new instance of a pixi stage
    this.stage = new PIXI.Stage(sceneBgColor);
    //create a renderer instance.
    var renderer = PIXI.autoDetectRenderer(sceneWidth, sceneHeight);
    var components = []; 
    	
    this.init = function() {

        if(typeof(canvasContainer) === 'string') {
            canvasContainer = document.getElementById(canvasContainer);
        }
        if(typeof(timeContainer) === 'string') {
            timeContainer = document.getElementById(timeContainer);
        }

        canvasContainer.appendChild(renderer.view);
        
        components.forEach(function(c){
    		c.init();
    	});
    };
    
    this.registerComponent = function(component) {
    	components.push(component);
    	this.stage.addChild(component.container);
    };

    this.refresh = function() {
    	components.forEach(function(c){
    		c.refresh();
    	});
        renderer.render(this.stage);
    };

    // Init page and intervals
    var refreshInterval;
    var refreshTimeInterval;
    var startTs;

    this.updateTime = function(){
        var nbSec = (Date.now() - startTs) / 1000;
        var hours = Math.floor( nbSec / 3600 ) % 24;
        var minutes = Math.floor( nbSec / 60 ) % 60;
        var seconds = Math.floor(nbSec % 60);
        var timeStr = (hours < 10 ? '0' + hours : hours) + ':' + (minutes < 10 ? '0' + minutes : minutes) + ':' + (seconds  < 10 ? '0' + seconds : seconds);
        timeContainer.innerHTML = timeStr;
    };

    this.start = function() {

        startTs = Date.now();
        if(!externalRefresh) {
            refreshInterval = setInterval(function() {_this.refresh();}, 1000/this.framerate);
        }
        refreshTimeInterval = setInterval(function() {_this.updateTime();}, 1000);
        
        components.forEach(function(c){
    		c.start();
    	});
    };

    this.stop = function() {
        if(!externalRefresh) {
            clearInterval(refreshInterval);
        }
        clearInterval(refreshTimeInterval);
        
        components.forEach(function(c){
    		c.stop();
    	});
    };


    this.log = function(m) {
        if(this.logger) {
            this.logger.log(m);
        }
    };


    return this;
}

module.exports = {
    StageView: StageView
};
