/**
* js/annotsRoll.js
*
* annotsRoll basic component
*
*/

'use strict';

var PIXI = require('pixi');
var _ = require('lodash');

var DEFAULT_ANNOT_COLOR = '#bababa';

var defaultAnnotStyles = {
    'label': { font: '16pt Arial Bold', fill: '#65A954', wordWrap: true},
    'text' : { font: '12pt Arial Regular', fill: '#444444', wordWrap: true},
    'user' : { font: '14pt Arial regular', fill: '#666666' },
};

var defaultOptions = {
    externalRefresh: false,
    defaultColor: DEFAULT_ANNOT_COLOR,
    annotStyles: defaultAnnotStyles,
    ignoreAnnots:false
};

function AnnotsRoll(options) {

//parentContainer, xInit, yInit, width, height, widthRoll, pixelsPerSecond, annotColors
    var _this = this;
    var opts = _(options).defaults(defaultOptions).value();


    this.container = new PIXI.DisplayObjectContainer();
    this.container.x = opts.xInit;
    this.container.y = opts.yInit;
    this.container.width = opts.width;

    this.height = opts.height;
    this.width = opts.width;
    this.widthRoll = opts.widthRoll;
    this.pixelsPerSecond = opts.pixelsPerSecond;
    this.annotColors = opts.annotColors;
    this.startTs = opts.startTs || Date.now();
    this.ignoreAnnots = opts.ignoreAnnots;

    var yInit = opts.yInit;
    var annotStyles = _(opts.annotStyles).defaults(defaultAnnotStyles).value();
    var marginX = 15;
    for(var style in annotStyles) {
    	if (annotStyles[style].wordWrap === true){
    		annotStyles[style].wordWrapWidth = this.widthRoll - marginX; 
    	}
    }
    var started = false;
    var ws = opts.ws;
    var externalRefresh = opts.externalRefresh;
    var stageView = opts.stageView;
    var waitInterval;
    var wait = 0;
    
    stageView.registerComponent(this);

    var isHidden = function(child) {
        // TODO: the origin point is an approximation. Should refine this
        var globalPos = child.toGlobal(new PIXI.Point(0,0));
        return ((globalPos.x + child.width) < 0) || ((globalPos.y + child.height) < 0) ;
    };

    this.addAnnots = function(data) {

        //var title = data.content.category.label;
        //var user = data.content.user;
        //Test cat and color
        //var colorAnnot = 0x65A954;
        var category = data.content.category.label,
            text     = data.content.text,
            user     = data.content.user,
            ts       = Date.parse(data.ts),
            color    = data.content.color || this.getColor(ts, data.content.category.code);

        this.addAnnot(category, text, user, color, ts);
    };

    this.getColor = function(ts, code) {
        var colorsDef;
        _(this.annotColors).eachRight(function(cdef) {
            if(cdef.ts < ts) {
                colorsDef = cdef.colors;
                return false;
            }
        });
        var resColor;
        if(colorsDef) {
            resColor = colorsDef[code];
        }
        if(!resColor) {
            resColor = colorsDef.defaultColor || DEFAULT_ANNOT_COLOR;
        }
        return resColor;
    }

    this.addAnnot = function(category, text, user, color, ts){

        var x = 0;
        var y = (ts-this.startTs) * this.pixelsPerSecond / 1000 + yInit;

        var colorHex = parseInt(color.replace(/^#/, ''), 16);

        if (wait === 0){
	        var graphics = new PIXI.Graphics()
	            .beginFill(colorHex)
	            .drawRect(x, y, 10, 3)
	            .endFill();
	
	        this.container.addChild(graphics);
	
	        var textHeight = 0;
	        var catLabel = new PIXI.Text(
	            category,
	            _(annotStyles.label).extend({fill: color}).value()
	        );
	        catLabel.x = x + marginX;
	        catLabel.y = y - 23;
	        this.container.addChild(catLabel);
	        textHeight += (catLabel.height - 23 + 2);
	
	        if(text) {
	            var catText = new PIXI.Text(text, annotStyles.text);
	            catText.x = x + marginX;
	            catText.y = y + textHeight;
	            this.container.addChild(catText);
	            textHeight += (catText.height + 2);
	        }
	
	        var catUser = new PIXI.Text(user, annotStyles.user);
	        catUser.x = x + marginX;
	        catUser.y = y + textHeight;
	        this.container.addChild(catUser);
	        textHeight += (catUser.height + 8);
	        
	        if (this.ignoreAnnots === true){
		        wait = textHeight / this.pixelsPerSecond;
		        waitInterval = setInterval(function() {_this.refreshWait();}, 1000);
	        }
        }

        this.addAnnotLine(colorHex, y);
    };

    this.addAnnotLine = function(color, y) {
        var x = this.widthRoll;


        var graphics = new PIXI.Graphics()
            .beginFill(color)
            .drawRect(x, y, this.width - x, 3)
            .endFill();

        this.container.addChild(graphics);
    };

    this.moveTo = function(diffTime){
    	this.container.y = Math.floor(diffTime*this.pixelsPerSecond);
    };

    this.move = this.refresh = function() {
        var diff = (this.startTs - Date.now())/1000;
        this.moveTo(diff);
    };
    
    this.refreshWait = function(){
    	wait -= 1;
    	if (wait < 0){
    		wait = 0;
    		clearInterval(waitInterval);
    	}
    };

    this.removePassedObjets = function(){
        var childrenToRemove = [];
        _(_this.container.children).forEach(function(child) {
            return typeof(child) === 'undefined' ||
                (isHidden(child) && childrenToRemove.push(child));
        });
        childrenToRemove.forEach(function(child) {
            _this.container.removeChild(child);
        });
    };

    this.init = function() {

        ws.message(function(data) {
            _this.addAnnots(data);
        });

    };
    

    this.start = function() {
        if(!started) {
            this.startTs = Date.now();
            started = true;
        }
        this.cleanInterval = setInterval(function () { _this.removePassedObjets(); }, 1000 * this.height / this.pixelsPerSecond );
        if(!externalRefresh) {
            this.refreshInterval = setInterval(function() {_this.move();}, 1000/this.framerate);
        }
    };

    this.stop = function() {
        clearInterval(this.cleanInterval);
        if(!externalRefresh) {
            clearInterval(this.refreshInterval);
        }
    };

}

module.exports = {
    AnnotsRoll: AnnotsRoll,
};
