/* Tracking module that listens to Renkan events */
/* globals Rkns */
/* eslint no-console: 0 */

'use strict';

Rkns.mtdcTracking = function (_renkan, _trackingWorker) {

    var _proj = _renkan.project;

    function prepareObjData(node) {
        var objData = node.toJSON(),
            p = node.get('project');
        objData.project = { id: p.get('id'), title: p.get('title') };
        objData.origin = node.get('origin', -1);
        return objData;
    }

    function registerNode(_node) {
        _node.on('remove', function (_n) {
            var nodeData = prepareObjData(_n);
            _trackingWorker.deleteNode(nodeData);
        });
        _node.on('change', function (_n) {
            var nodeData = prepareObjData(_n);
            var dataChanged = _n.changed;
            var previousData = Rkns._.mapValues(_n.changed, function (v, k) {
                return _n.previous(k);
            });
            _trackingWorker.updateNode(nodeData, dataChanged, previousData);
        });
    }

    function registerEdge(_edge) {
        _edge.on('remove', function (_e) {
            var edgeData = prepareObjData(_e);
            _trackingWorker.deleteEdge(edgeData);
        });
        _edge.on('change', function (_e) {
            var edgeData = prepareObjData(_e);
            var dataChanged = _e.changed;
            var previousData = Rkns._.mapValues(_e.changed, function (v, k) {
                return _e.previous(k);
            });
            _trackingWorker.updateEdge(edgeData, dataChanged, previousData);
        });
    }

    function registerView(_view) {
        _view.on('remove', function (_v) {
            var viewData = prepareObjData(_v);
            _trackingWorker.deleteView(viewData);
        });
        _view.on('change', function (_v) {
            var viewData = prepareObjData(_v);
            var dataChanged = _v.changed;
            var previousData = Rkns._.mapValues(_v.changed, function (v, k) {
                return _v.previous(k);
            });
            _trackingWorker.updateView(viewData, dataChanged, previousData);
        });
    }

    _proj.on('change:loadingStatus', function (_p) {
        if (_p.get('loadingStatus')) {
            return;
        }
        // -> track open for editing

        _p.on('add:nodes', function (_node) {
            var nodeData = prepareObjData(_node);
            registerNode(_node);
            _trackingWorker.addNode(nodeData);
        });

        _p.on('add:edges', function (_edge) {
            var edgeData = prepareObjData(_edge);
            registerEdge(_edge);
            _trackingWorker.addEdge(edgeData);
        });

        _p.on('add:views', function (_view) {
            var viewData = prepareObjData(_view);
            registerView(_view);
            _trackingWorker.addView(viewData);
        });

        _proj.get('nodes').each(function (_node) {
            registerNode(_node);
        });

        _proj.get('edges').each(function (_edge) {
            registerEdge(_edge);
        });

        _proj.get('views').each(function (_view) {
            registerView(_view);
        });
    });
};