import requests, logging

from datetime import timedelta

from django.core.exceptions import PermissionDenied
from django.core.urlresolvers import reverse
from django.contrib.auth.models import Permission
from django.contrib.auth import get_user_model
from django.http import HttpResponseRedirect
from django.utils import timezone

from allauth.socialaccount.providers.oauth2.views import (OAuth2Adapter,
                                                          OAuth2View,
                                                          OAuth2LoginView,
                                                          OAuth2CallbackView)
from allauth.socialaccount.providers.oauth2.client import (OAuth2Client,
                                                           OAuth2Error)

from allauth.socialaccount.helpers import complete_social_login, render_authentication_error
from allauth.socialaccount.models import SocialToken, SocialLogin
from allauth.account import app_settings
from allauth.account.utils import perform_login
from allauth.utils import build_absolute_uri, get_request_param
from allauth.socialaccount.providers.base import AuthAction, AuthError
from allauth.socialaccount.adapter import DefaultSocialAccountAdapter
from django.conf import settings
from urllib.parse import urlparse

from .provider import MtdcProvider

logger = logging.getLogger(__name__)

class MtdcOAuth2Adapter(OAuth2Adapter, DefaultSocialAccountAdapter):
    provider_id = MtdcProvider.id
    supports_state = False
    
    oauth_base_url = ""
    access_token_url = ""
    authorize_url = ""
    profile_url = ""
    
    def __init__(self, request=None):
        if request:
            logger.debug("AUTHORIZATION CODE AUTH: init adapter")
            if request.session.get("OAUTH_CONTEXT_BASE_URL", None) is None:
                logger.debug("AUTHORIZATION CODE AUTH: no context in session, storing context")
                request.session["OAUTH_CONTEXT_BASE_URL"] = request.GET.get("context", None)
                logger.debug("AUTHORIZATION CODE AUTH: context queryarg is %r", request.GET.get("context", None))
            self.oauth_base_url = request.session.get("OAUTH_CONTEXT_BASE_URL", None)
            logger.debug("AUTHORIZATION CODE AUTH: context is %r", self.oauth_base_url)
            self.access_token_url = self.oauth_base_url + settings.MTDC_ACCESS_TOKEN_URL
            self.authorize_url = self.oauth_base_url + settings.MTDC_AUTHORIZE_URL
            self.profile_url = self.oauth_base_url + settings.MTDC_PROFILE_URL
    
    def pre_social_login(self, request, sociallogin):
        try:
            logger.debug("AUTHORIZATION CODE AUTH: login almost complete, checking if user %r exists", sociallogin.account.extra_data.get('external_id', 'NO_ID'))
            user = get_user_model().objects.get(external_id=sociallogin.account.extra_data.get('external_id', ''))  # if user exists, connect the account to the existing account and login
            logger.debug("AUTHORIZATION CODE AUTH: user %r exists, connecting to existing account", sociallogin.account.extra_data.get('external_id', 'NO_ID'))
            sociallogin.state['process'] = 'connect'            
            perform_login(request, user, 'none')
        except get_user_model().DoesNotExist:
            logger.debug("AUTHORIZATION CODE AUTH: user %r does not exist", sociallogin.account.extra_data.get('external_id', 'NO_ID'))
    
    def get_login_redirect_url(self, request):
        return super(MtdcOAuth2Adapter, self).get_login_redirect_url(self, request)
    
    def new_user(self, request, sociallogin):
        if 'external_id' in sociallogin.account.extra_data:
            logger.debug("AUTHORIZATION CODE AUTH: checking if user %r exists to populate sociallogin", sociallogin.account.extra_data.get('external_id', 'NO_ID'))
            user_queryset = get_user_model().objects.filter(external_id=sociallogin.account.extra_data.get('external_id', ''), username=sociallogin.account.extra_data['username'])
            if user_queryset.exists():
                logger.debug("AUTHORIZATION CODE AUTH: user %r exists", sociallogin.account.extra_data.get('external_id', 'NO_ID'))
                user = user_queryset.first()
            else:
                logger.debug("AUTHORIZATION CODE AUTH: user %r does not exist, creating new user and populating", sociallogin.account.extra_data.get('external_id', 'NO_ID'))
                user = get_user_model()()
            return user
        else:
            logger.debug("AUTHORIZATION CODE AUTH: no external_id in extra data")
            return get_user_model()()
        
    def populate_user(self,
                      request,
                      sociallogin,
                      data):
        user = sociallogin.user
        user.username = sociallogin.account.extra_data.get('username', '')
        user.external_id = sociallogin.account.extra_data.get('external_id', '')
        user.uai = sociallogin.account.extra_data.get('uai', '')
        user.first_name = sociallogin.account.extra_data.get('first_name', '')
        user.last_name = sociallogin.account.extra_data.get('last_name', '')
        user.save()
        return user
    
    def complete_login(self, request, app, token, **kwargs):
        logger.debug("AUTHORIZATION CODE AUTH: complete_login: getting profile info")
        resp = requests.get(self.profile_url,
                            params={'access_token': token.token})
        extra_data = resp.json()
        
        logger.debug("AUTHORIZATION CODE AUTH: response extra_data: %r ", extra_data)
        return self.get_provider().sociallogin_from_response(request,
                                                             extra_data)

class MtdcOAuth2View(OAuth2View):
    @classmethod
    def adapter_view(cls, adapter):
        def view(request, *args, **kwargs):
            self = cls()
            self.request = request
            self.adapter = adapter(request)
            return self.dispatch(request, *args, **kwargs)
        return view

class MtdcOAuth2LoginView(MtdcOAuth2View, OAuth2LoginView):
    def dispatch(self, request):
        logger.debug("AUTHORIZATION CODE AUTH: dispatching LoginView")
        return super(MtdcOAuth2LoginView, self).dispatch(request)

class MtdcOAuth2CallbackView(MtdcOAuth2View, OAuth2CallbackView):
    def dispatch(self, request):
        logger.debug("AUTHORIZATION CODE AUTH: dispatching CallbackView")
        return super(MtdcOAuth2CallbackView, self).dispatch(request)


oauth2_login = MtdcOAuth2LoginView.adapter_view(MtdcOAuth2Adapter)
oauth2_callback = MtdcOAuth2CallbackView.adapter_view(MtdcOAuth2Adapter)