import datetime
import json
import logging
import pytz
import string

from django.conf import settings
from django.utils import six

from .tasks import send_tracking_data

logger = logging.getLogger(__name__)

def get_user_name(user):
    if not user:
        return 'n/a'
    if isinstance(user, six.string_types):
        return user
    try:
        if user.is_authenticated() and user.external_id:
            return str(user.external_id)
        else:
            return 'Anonymous'
    except:
        return 'anonymous'


def get_base_message(verb, renkan_id, current_user, registration = None):
    #create
    #open-read
    #open-edit
    #close
    #delete
    #update
    timestamp = datetime.datetime.utcnow().replace(tzinfo=pytz.utc)

    verbNode = {
        "create" : {
            "id": 'http://activitystrea.ms/schema/1.0/create',
            "display": { 'fr-FR': 'a créé' }
        },
        "update": {
            "id": 'http://activitystrea.ms/schema/1.0/update',
            "display": { 'fr-FR': 'a modifié' }
        },
        "delete": {
            "id": 'http://activitystrea.ms/schema/1.0/delete',
            "display": { 'fr-FR': 'a supprimé' }
        },
        "close": {
            "id": 'http://activitystrea.ms/schema/1.0/close',
            "display": { 'fr-FR': 'a fermé'}
        },
        "open_read": {
            "id": 'http://id.tincanapi.com/verb/viewed',
            "display": { 'fr-FR': 'a vu'}
        },
        "open_edit": {
            "id" : 'http://activitystrea.ms/schema/1.0/access',
            "display": { 'fr-FR': "a édité" }
        }
    }[verb];
    msg = {
        'actor': {
            'objectType': 'Agent',
            'name': get_user_name(current_user),
            'account': {
                'homePage': 'https://www.metaeducation.fr/Utilisateurs/',
                'name': get_user_name(current_user)
            }
        },
        'verb': verbNode,
        'object': {
            'objectType': 'Activity',
            'id': get_renkan_uri(renkan_id)
        },
        'context': {
            'extensions': {
                'http://liris.renkantracking.org/application': 'Outil carte mentale'
            }
        },
        'timestamp': timestamp.isoformat()
    };
    if registration:
        msg['context']['registration'] = registration

    return msg

def get_renkan_uri(renkan_id):
    template = string.Template(settings.TRACKING_URI_TEMPLATES.get('renkan', 'urn:mtdc:renkan:renkan:${renkan_id}'))
    return template.safe_substitute({'renkan_id': renkan_id})


def send_close_renkan(renkan, current_user, registration):
    msg = get_base_message('close', renkan.renkan_guid, current_user, registration)

    msg['object'] = {
        **(msg['object']),
        **{
            "definition": {
                "name": {
                    'fr-FR': renkan.title
                },
                "type": "http://www.w3.org/ns/activitystreams#Renkan",
                "extensions": {
                    'http://www.w3.org/ns/activitystreams#Data': json.loads(renkan.content),
                }
            }
        }
    }

    send_tracking_data.delay(msg)


def send_open_read_renkan(renkan, current_user, registration):
    msg = get_base_message('open_read', renkan.renkan_guid, current_user, registration)

    msg['object'] = {
        **(msg['object']),
        **{
            "definition": {
                "name": {
                    'fr-FR': renkan.title
                },
                "type": "http://www.w3.org/ns/activitystreams#Renkan",
                "extensions": {
                    'http://www.w3.org/ns/activitystreams#Data': json.loads(renkan.content),
                }
            }
        }
    }

    send_tracking_data.delay(msg)

def send_open_edit_renkan(renkan, current_user, registration):
    msg = get_base_message('open_edit', renkan.renkan_guid, current_user, registration)

    msg['object'] = {
        **(msg.get('object', {})),
        **{
            "definition": {
                "name": {
                    'fr-FR': renkan.title
                },
                "type": "http://www.w3.org/ns/activitystreams#Renkan",
                "extensions": {
                    'http://www.w3.org/ns/activitystreams#Data': json.loads(renkan.content),
                }
            }
        }
    }

    send_tracking_data.delay(msg)

def send_delete_renkan(renkan, current_user):
    content = json.loads(renkan.content)
    title = renkan.title
    renkan_guid = renkan.renkan_guid

    msg = get_base_message('delete', renkan_guid, current_user)
    msg['object'] = {
        **(msg.get('object',{})),
        **{
            "definition": {
                "name": {
                    'fr-FR': title
                },
                "type": "http://www.w3.org/ns/activitystreams#Renkan",
                "extensions": {
                    'http://www.w3.org/ns/activitystreams#Data': content,
                }
            }
        }
    }
    send_tracking_data.delay(msg)

# This must be called with a revision object containing the new content BEFORE the object save
def send_update_renkan(revision, current_user):
    renkan = revision.parent_renkan
    previousContent = json.loads(renkan.content)
    content = json.loads(revision.content)
    title = renkan.title
    renkan_guid = renkan.renkan_guid
    msg = get_base_message('update', renkan_guid, current_user)
    msg['object'] = {
        **(msg.get('object',{})),
        **{
            "definition": {
                "name": {
                    'fr-FR': title
                },
                "type": "http://www.w3.org/ns/activitystreams#Renkan",
                "extensions": {
                    'http://www.w3.org/ns/activitystreams#Data': content,
                    'http://www.w3.org/ns/activitystreams#previousData': previousContent,
                }
            }
        }
    }
    send_tracking_data.delay(msg)


def send_create_renkan(renkan, current_user):
    msg = get_base_message('create', renkan.renkan_guid, current_user)
    msg['object'] = {
        **(msg.get('object', {})),
        **{
            "definition": {
                "name": {
                    'fr-FR': renkan.title
                },
                "type": "http://www.w3.org/ns/activitystreams#Renkan",
                "extensions": {
                    'http://www.w3.org/ns/activitystreams#Data': json.loads(renkan.content),
                }
            }
        }
    }
    if renkan.source_revision:
        msg['context']['extensions'] = {
            **msg.get('context', {}).get('extensions', {}),
            **{
                'http://liris.renkantracking.org/fromCreate': {'renkan': renkan.source_revision.parent_renkan.renkan_guid, 'revision': renkan.source_revision.revision_guid}
            }
        }
    send_tracking_data.delay(msg)
