import logging
import uuid

from django.db.models.signals import pre_delete, post_save
from django.utils.functional import curry

from renkanmanager.models import Renkan, Revision

from . import send_delete_renkan, send_create_renkan, send_update_renkan

logger = logging.getLogger(__name__)

# Inspired by https://github.com/Atomidata/django-audit-log/blob/master/audit_log/middleware.py

class TrackingMiddleware(object):
    def process_request(self, request):
        if not request.method in ('GET', 'HEAD', 'OPTIONS', 'TRACE'):
            if hasattr(request, 'user') and request.user.is_authenticated():
                user = request.user.external_id
            else:
                user = None

            request.renkan_request_id = uuid.uuid4()

            pre_delete.connect(curry(self.renkan_delete, user), sender=Renkan, dispatch_uid = (self.__class__,  request.renkan_request_id,), weak=False)
            post_save.connect(curry(self.revision_save, user), sender=Revision, dispatch_uid = (self.__class__,  request.renkan_request_id,), weak=False)

    def process_response(self, request, response):
        if hasattr(request, 'renkan_request_id'):
            pre_delete.disconnect(dispatch_uid =  (self.__class__,  request.renkan_request_id,))
            post_save.disconnect(dispatch_uid =  (self.__class__,  request.renkan_request_id,))
        return response

    def process_exception(self, request, exception):
        if hasattr(request, 'renkan_request_id'):
            pre_delete.disconnect(dispatch_uid =  (self.__class__,  request.renkan_request_id,))
            post_save.disconnect(dispatch_uid =  (self.__class__,  request.renkan_request_id,))
        return None

    def process_view(self, request, view_func, view_args, view_kwargs):
        if view_func.__name__ == 'RenkanDetail' and view_func.__module__ == 'renkanmanager.api.views' and request.method == 'PUT' and 'renkan_guid' in view_kwargs:
            if hasattr(request, 'user') and request.user.is_authenticated():
                user = request.user.external_id
            else:
                user = None
            send_update_renkan(None, user, request.body.decode('utf-8'))
        return None


    def renkan_delete(self, user, sender, **kwargs):
        renkan = kwargs.get('instance', None)
        if not renkan:
            return
        send_delete_renkan(renkan, user)

    def revision_save(self, user, sender, **kwargs):
        revision = kwargs.get('instance', None)
        if not revision:
            return
        if kwargs.get('created', False) and revision.parent_renkan.revision_count <= 1:
            send_create_renkan(revision.parent_renkan, user)
