import logging, json, uuid

from django import forms
from django.conf import settings
from django.core.urlresolvers import reverse
from django.http import HttpResponse, HttpResponseRedirect
from django.shortcuts import render, get_object_or_404
from django.views.generic import TemplateView, View

from renkanmanager.api.views import RenkanDetail
from renkanmanager.models import Renkan
from renkanmanager.serializers import RenkanSerializer

from metaeducation.tracking import send_open_read_renkan, send_open_edit_renkan
from metaeducation import __version__

logger = logging.getLogger(__name__)

class ListRenkansView(View):
    template_name = "renkan_list.html"

    def get(self, request):
        renkans = Renkan.objects.filter(creator=request.user)
        return render(request, self.template_name, {'renkans': renkans, 'version': __version__})

    def post(self, request):
        create_data = {
            "title" : request.POST.get("title", "Untitled Renkan")
        }

        serializer = RenkanSerializer(data=create_data)
        if serializer.is_valid():
            serializer.save(creator=request.user)
        return HttpResponseRedirect(request.META.get('HTTP_REFERER', settings.BASE_URL))

class NewRenkanView(View):

    def get(self, request):
        create_data = {
            "title" : "Untitled Renkan"
        }
        serializer = RenkanSerializer(data=create_data)
        if serializer.is_valid():
            new_renkan = serializer.save(creator=request.user)
            return HttpResponseRedirect(reverse("front_edit_renkan", kwargs={"renkan_guid": new_renkan.renkan_guid}))


class ViewRenkanView(TemplateView):
    template_name = "renkan_view.html"
    def get_context_data(self, **kwargs):
        context = super(ViewRenkanView, self).get_context_data(**kwargs)
        context['registration'] = str(uuid.uuid4())
        return context

    def get(self, request, *args, **kwargs):
        renkan = get_object_or_404(Renkan.objects.select_related('current_revision'), renkan_guid=kwargs['renkan_guid'])
        context = self.get_context_data(**kwargs)
        send_open_read_renkan(renkan, str(getattr(request.user,'external_id', 'n/a')), str(context['registration']))
        return self.render_to_response(context)



class EditRenkanView(TemplateView):
    template_name = "renkan_edit.html"
    def get_context_data(self, **kwargs):
        context = super(EditRenkanView, self).get_context_data(**kwargs)
        context['registration'] = str(uuid.uuid4())
        return context

    def get(self, request, *args, **kwargs):
        renkan = get_object_or_404(Renkan.objects.select_related('current_revision'), renkan_guid=kwargs['renkan_guid'])
        context = self.get_context_data(**kwargs)
        send_open_edit_renkan(renkan, str(getattr(request.user,'external_id', 'n/a')), str(context['registration']))
        return self.render_to_response(context)

class DeleteForm(forms.Form):
    renkan_guid = forms.UUIDField()

class DeleteRenkanView(View):

    def post(self, request):
        form = DeleteForm(request.POST)
        if form.is_valid():
            renkan_guid = form.cleaned_data['renkan_guid']
            logger.debug("DELETE FORM VALID %r", form.cleaned_data)
            request.method = "DELETE"
            delete_response = RenkanDetail.as_view()(request, str(renkan_guid))
            return HttpResponseRedirect(request.META.get('HTTP_REFERER', settings.BASE_URL))
        return HttpResponse('field renkan_guid unknown or bad format', status=400)
