from functools import partial

import logging

from django.db.models.signals import pre_delete, post_save, pre_save

from renkanmanager.models import Renkan, Revision
from . import send_delete_renkan, send_create_renkan, send_update_renkan

logger = logging.getLogger(__name__)

# Inspired by https://github.com/Atomidata/django-audit-log/blob/master/audit_log/middleware.py

def register_handlers(user, uid):
    pre_delete.connect(partial(renkan_delete, user), sender = Renkan, dispatch_uid = uid, weak = False)
    post_save.connect(partial(revision_post_save, user), sender = Revision, dispatch_uid = uid, weak = False)

def register_pre_save_handlers(user, uid):
    pre_save.connect(partial(revision_pre_save, user), sender = Revision, dispatch_uid = uid, weak = False)


def unregister_handlers(uid):
    pre_delete.disconnect(dispatch_uid = uid, sender = Renkan)
    post_save.disconnect(dispatch_uid = uid, sender = Revision)
    pre_save.disconnect(dispatch_uid = uid, sender = Revision)

def change_handlers_user(user, uid):
    if pre_delete.disconnect(dispatch_uid = uid, sender = Renkan):
        pre_delete.connect(partial(renkan_delete, user), sender = Renkan, dispatch_uid = uid, weak = False)
    if post_save.disconnect(dispatch_uid = uid, sender = Revision):
        post_save.connect(partial(revision_post_save, user), sender = Revision, dispatch_uid = uid, weak = False)
    if pre_save.disconnect(dispatch_uid = uid, sender = Revision):
        pre_save.connect(partial(revision_pre_save, user), sender = Revision, dispatch_uid = uid, weak = False)


def renkan_delete(user, sender, **kwargs):
    renkan = kwargs.get('instance', None)
    if not renkan:
        return
    send_delete_renkan(renkan, user)

def revision_post_save(user, sender, **kwargs):
    revision = kwargs.get('instance', None)

    if not revision:
        return
    if kwargs.get('created', False) and revision.parent_renkan.revision_count <= 1:
        send_create_renkan(revision.parent_renkan, user)

#This handler is used and registered in a single case, i.e. PUT on renkanmanager.api.views.RenkanDetail
def revision_pre_save(user, sender, **kwargs):
    revision = kwargs.get('instance', None)

    if not revision:
        return

    # we look for renkan update : i.e. creation of a revision with a revision count > 0 or simple revision update
    send_update_renkan(revision, user)
