import requests

from datetime import timedelta

from django.core.exceptions import PermissionDenied
from django.core.urlresolvers import reverse
from django.contrib.auth.models import Permission
from django.contrib.auth import get_user_model
from django.http import HttpResponseRedirect
from django.utils import timezone

from allauth.socialaccount.providers.oauth2.views import (OAuth2Adapter,
                                                          OAuth2View,
                                                          OAuth2LoginView,
                                                          OAuth2CallbackView)
from allauth.socialaccount.providers.oauth2.client import (OAuth2Client,
                                                           OAuth2Error)

from allauth.socialaccount.helpers import complete_social_login, render_authentication_error
from allauth.socialaccount.models import SocialToken, SocialLogin
from allauth.account import app_settings
from allauth.account.utils import perform_login
from allauth.utils import build_absolute_uri, get_request_param
from allauth.socialaccount.providers.base import AuthAction, AuthError
from allauth.socialaccount.adapter import DefaultSocialAccountAdapter
from django.conf import settings
from urllib.parse import urlparse

from .provider import MtdcProvider


class MtdcOAuth2Adapter(OAuth2Adapter, DefaultSocialAccountAdapter):
    provider_id = MtdcProvider.id
    supports_state = False
    
    oauth_base_url = ""
    access_token_url = ""
    authorize_url = ""
    profile_url = ""
    
    def __init__(self, request=None):
        if request:
            if request.session.get("OAUTH_CONTEXT_BASE_URL", None) is None:
                request.session["OAUTH_CONTEXT_BASE_URL"] = request.GET.get("context", None)
            self.oauth_base_url = request.session.get("OAUTH_CONTEXT_BASE_URL", None)
            self.access_token_url = self.oauth_base_url + settings.MTDC_ACCESS_TOKEN_URL
            self.authorize_url = self.oauth_base_url + settings.MTDC_AUTHORIZE_URL
            self.profile_url = self.oauth_base_url + settings.MTDC_PROFILE_URL
    
    def pre_social_login(self, request, sociallogin):
        user = sociallogin.user   
        try:
            user = get_user_model().objects.get(username=user.username)  # if user exists, connect the account to the existing account and login
            sociallogin.state['process'] = 'connect'                
            perform_login(request, user, 'none')
        except get_user_model().DoesNotExist:
            pass
    
    def get_login_redirect_url(self, request):
        return super(MtdcOAuth2Adapter, self).get_login_redirect_url(self, request)
    
    def new_user(self, request, sociallogin):
        if 'username' in sociallogin.account.extra_data:
            user_queryset = get_user_model().objects.filter(username=sociallogin.account.extra_data['username'])
            if user_queryset.exists():
                user = user_queryset.first()
            else:
                user = get_user_model()()
                user.username = sociallogin.account.extra_data.get('username', '')
            return user
        else:
            return get_user_model()()
        
    def populate_user(self,
                      request,
                      sociallogin,
                      data):
        username = data.get('username')
        user = sociallogin.user
        user.username = username
        user.save()
        add_permission = Permission.objects.get(codename="add_renkan")
        user.user_permissions.add(add_permission)
        return user
    
    def complete_login(self, request, app, token, **kwargs):
        resp = requests.get(self.profile_url,
                            params={'access_token': token.token})
        print(resp.text)
        extra_data = resp.json()
        if request.session.get("OAUTH_CONTEXT_BASE_URL", None) is not None:
            del request.session["OAUTH_CONTEXT_BASE_URL"]
        return self.get_provider().sociallogin_from_response(request,
                                                             extra_data)

class MtdcOAuth2View(OAuth2View):
    @classmethod
    def adapter_view(cls, adapter):
        def view(request, *args, **kwargs):
            self = cls()
            self.request = request
            self.adapter = adapter(request)
            return self.dispatch(request, *args, **kwargs)
        return view

class MtdcOAuth2LoginView(MtdcOAuth2View, OAuth2LoginView):
    def dispatch(self, request):
        return super(MtdcOAuth2LoginView, self).dispatch(request)

class MtdcOAuth2CallbackView(MtdcOAuth2View, OAuth2CallbackView):
    def dispatch(self, request):
        return super(MtdcOAuth2CallbackView, self).dispatch(request)


oauth2_login = MtdcOAuth2LoginView.adapter_view(MtdcOAuth2Adapter)
oauth2_callback = MtdcOAuth2CallbackView.adapter_view(MtdcOAuth2Adapter)