import requests, json, sys, logging

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.auth.signals import user_logged_in, user_logged_out
from django.core.urlresolvers import reverse
from django.db.models.signals import post_save
from django.dispatch import receiver
from guardian.shortcuts import assign_perm


logger = logging.getLogger(__name__)


def reference_created_renkan(sender, instance, created, **kwargs):
    from renkanmanager.models import Revision
    if not settings.MTDC_CLIENT_CREDENTIALS_TOKEN_URL or not settings.MTDC_REFERENCE_RESOURCE_BASE_URL:
        return
    if created and sender == Revision and not instance.parent_renkan.source_revision_id and instance.parent_renkan.revision_count <= 1:
        token_response = requests.post(
            settings.MTDC_CLIENT_CREDENTIALS_TOKEN_URL+"?grant_type=client_credentials",
            data = {},
            headers = {
                "Authorization": "Basic %s" % settings.MTDC_AUTH_CODE
            }
        )
        if token_response.status_code == 200:
            logger.debug("REFERENCING RENKAN: token response is 200")
            token = json.loads(token_response.text)['access_token']
            post_data = json.dumps({
                "id": str(instance.parent_renkan.renkan_guid),
                "idOutil": settings.RENKAN_TOOL_ID,
                "idUser": instance.creator.external_id,
                "pf": settings.ITOP_PF_CODE,
                "uai": instance.creator.uai,
                "redirect_uri": settings.OAUTH_REDIRECT_URI,
                "title": instance.title
            })
            resource_ws_url = settings.MTDC_REFERENCE_RESOURCE_BASE_URL# + str(instance.renkan_guid)
            logger.debug("REFERENCING RENKAN: sending reference request to: POST %r", resource_ws_url)
            logger.debug("REFERENCING RENKAN: post_data: %r", post_data)
            reference_response = requests.post(
                resource_ws_url,
                data = post_data,
                headers = {
                    "Authorization": "Bearer %s" % token,
                    "content-type": "application/json"
                }
            )
            logger.debug("REFERENCING RENKAN: response is %r", reference_response.status_code)

def assign_objects_permissions(sender, instance, created, **kwargs):
    from renkanmanager.models import Renkan, Revision, Workspace
    if sender == Renkan and created:
        creator = instance.creator
        logger.debug("SIGNALS: RENKAN WAS CREATED, ASSIGNING PERMS TO USER %r FOR RENKAN %r", instance.creator, instance.renkan_guid)
        assign_perm('renkanmanager.view_renkan', creator, instance)
        assign_perm('renkanamanger.change_renkan', creator, instance)
        assign_perm('renkanmanager.delete_renkan', creator, instance)
    if sender == Revision and created:
        creator = instance.creator
        logger.debug("SIGNALS: REVISION WAS CREATED, ASSIGNING PERMS TO USER %r", instance.creator)
        assign_perm('renkanmanager.view_revision', creator, instance)
        assign_perm('renkanmanager.delete_revision', creator, instance)
    if sender == Workspace and created:
        creator = instance.creator
        logger.debug("SIGNALS: WORKSPACE WAS CREATED, ASSIGNING PERMS TO USER %r", instance.creator)
        assign_perm('renkanmanager.view_workspace', creator, instance)
        assign_perm('renkanmanager.change_workspace', creator, instance)
        assign_perm('renkanmanager.delete_workspace', creator, instance)

def assign_user_permissions(sender, instance, created, **kwargs):
    if sender == get_user_model() and created:
        logger.debug("SIGNALS: USER %r WAS CREATED, ASSIGNING MODEL PERMISSIONS", instance)
        assign_perm('renkanmanager.view_renkan', instance)
        assign_perm('renkanmanager.add_renkan', instance)
        assign_perm('renkanmanager.change_renkan', instance)
        assign_perm('renkanmanager.delete_renkan', instance)

def log_user_logged_in(sender, request, user, **kwargs):
    logger.debug("LOGGING IN: user logged in: %r", user)

def log_user_logged_out(sender, request, user, **kwargs):
    logger.debug("LOGGING OUT: DJANGO: user logged out: %r", user)

def token_updated(sender, instance, created, **kwargs):
    from allauth.socialaccount.models import SocialToken
    if sender == SocialToken:
        logger.debug("TOKEN: SocialToken saved")
        logger.debug("TOKEN: Token created: %r", created)
    
if not 'test' in sys.argv:
    post_save.connect(reference_created_renkan)
    post_save.connect(token_updated)
    post_save.connect(assign_objects_permissions)
    post_save.connect(assign_user_permissions)
    user_logged_in.connect(log_user_logged_in)
    user_logged_out.connect(log_user_logged_out)