from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.core.paginator import Paginator, InvalidPage, EmptyPage
from django.shortcuts import redirect
from django.utils.decorators import method_decorator
from django.views.decorators.cache import never_cache
from django.views.generic.base import View, TemplateResponseMixin
from ldt.ldt_utils.models import Project, Content
from ldt.ldt_utils.views.content import write_content_base
from ldt.ldt_utils.utils import generate_uuid
from metadatacomposer.forms import ImageUploadModelForm
from metadatacomposer.models import Image


import logging #@UnresolvedImport
logger = logging.getLogger(__name__)


class MetadataComposerContextView(View):
    
    branding = "iri"
    
    def get_context_dict(self, request):
        context = {}
        context["branding"] = self.branding
        context["top_header_css"] = "metadatacomposer/%s/composer_header_css.html" % self.branding
        context["top_header_partial"] = "metadatacomposer/%s/composer_header.html" % self.branding
        return context



class MetadataComposerHome(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_home.html"
    
    @method_decorator(login_required)
    @method_decorator(never_cache)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerHome, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        
        projects = Project.safe_objects.filter(owner=request.user).order_by('-modification_date')[:6]
        images = Image.objects.order_by('-modification_date')[:6]
        contents = Content.safe_objects.order_by('-update_date')[:6]
        
        context = self.get_context_dict(request)
        context.update({"projects":projects, "images": images, "contents": contents})
        return self.render_to_response(context)



class MetadataComposerProjectList(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_project_list.html"
    
    @method_decorator(login_required)
    @method_decorator(never_cache)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerProjectList, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        page = request.GET.get("page") or 1
        
        projects = Project.safe_objects.prefetch_related("contents").order_by('-modification_date').filter(owner=request.user)
        nb = getattr(settings, 'METADATACOMPOSER_ELEMENTS_PER_PAGE', 9)
        if page=="x":
            nb = projects.count()
        
        paginator = Paginator(projects, nb)
        try:
            results = paginator.page(page)
        except (EmptyPage, InvalidPage):
            results = paginator.page(paginator.num_pages)
        
        context = self.get_context_dict(request)
        context.update({"results":results})
        return self.render_to_response(context)



class MetadataComposerResourceList(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_resource_list.html"
    
    @method_decorator(login_required)
    @method_decorator(never_cache)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerResourceList, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        
        # We get the first page of images
        image_results = get_images(1)
        # We get the first contents page and theirs projects
        content_results = get_contents_and_projects(1, request.user)
        
        context = self.get_context_dict(request)
        context.update({"image_results":image_results, "content_results":content_results})
        return self.render_to_response(context)



class MetadataComposerImagePagination(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        if self.template_name and self.template_name!="":
            return self.template_name
        else:
            return "partial/resource_image_list.html"
    
    @method_decorator(login_required)
    @method_decorator(never_cache)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerImagePagination, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        page = request.GET.get("page") or 1
        mode = request.GET.get("mode") or "resource"
        # Get current contents page and theirs projects
        image_results = get_images(page)
        
        if mode=="library":
            self.template_name = "partial/library_image_list.html"
        
        context = self.get_context_dict(request)
        context.update({"image_results":image_results})
        return self.render_to_response(context)



class MetadataComposerContentPagination(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "partial/resource_content_list.html"
    
    @method_decorator(login_required)
    @method_decorator(never_cache)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerContentPagination, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        page = request.GET.get("page") or 1
        # Get current contents page and theirs projects
        content_results = get_contents_and_projects(page, request.user)
        
        context = self.get_context_dict(request)
        context.update({"content_results":content_results})
        return self.render_to_response(context)



def get_images(page):
    
    # We get the current's page images
    images = Image.objects.order_by('-modification_date')
    #nb = getattr(settings, 'METADATACOMPOSER_ELEMENTS_PER_PAGE', 9)
    nb = 2
    if page=="x":
        nb = images.count()
    paginator = Paginator(images, nb)
    try:
        results = paginator.page(page)
    except (EmptyPage, InvalidPage):
        results = paginator.page(paginator.num_pages)
    
    return results



def get_contents_and_projects(page, user):
    
    # We get the current's page contents
    # prefetch_related("project_set") is unfortunately useless because we have to filter the project queryset later
    contents = Content.safe_objects.order_by('-update_date')
    nb = getattr(settings, 'METADATACOMPOSER_ELEMENTS_PER_PAGE', 9)
    if page=="x":
        nb = contents.count()
    paginator = Paginator(contents, nb)
    try:
        results = paginator.page(page)
    except (EmptyPage, InvalidPage):
        results = paginator.page(paginator.num_pages)
    
    # We add the user's projects for each content
    results_object_list = []
    for content in results.object_list:
        # We filter the content's projects with the user's ones
        projects = content.project_set.all().filter(owner=user)
        results_object_list.append({"content":content, "projects":projects})
    
    results.object_list = results_object_list
    return results



class MetadataComposerModalVideo(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_modal_video.html"
    
    @method_decorator(login_required)
    @method_decorator(never_cache)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerModalVideo, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", iri_id=None, **kwargs):
        self.branding = branding
        
        # Generate fake id to validate ContentForm
        if not iri_id:
            iri_id = generate_uuid()
            content = None
        else:
            content = Content.safe_objects.select_related("media_obj").get(iri_id=iri_id)
        
        context = self.get_context_dict(request)
        context.update({"iri_id":iri_id, "content":content})
        return self.render_to_response(context)
    
    def post(self, request, branding="iri", iri_id=None, **kwargs):
        self.branding = branding
        
        # We create the media
        content_form, media_form, picture_form, form_status, _, current_front_project, e, traceback = write_content_base(request, iri_id)
        # And test creation
        if (content_form == False and media_form == False and picture_form == False and form_status == False and current_front_project == False):
            #message=_("An error occurred - Please try again or contact webmaster")
            #title = _("Error")
            raise e, None, traceback
        return redirect(request.META['HTTP_REFERER'])



class MetadataComposerModalImage(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_modal_image.html"
    
    @method_decorator(login_required)
    @method_decorator(never_cache)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerModalImage, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        image_pk = request.GET.get("image") or None
        
        # Add form
        if image_pk:
            form = ImageUploadModelForm(instance=Image.objects.get(pk=image_pk))
        else:
            form = ImageUploadModelForm()
        
        context = self.get_context_dict(request)
        context.update({"form":form, "image_pk":image_pk})
        return self.render_to_response(context)
    
    def post(self, request, branding="iri", **kwargs):
        self.branding = branding
        
        # Check form
        if 'image_pk' in request.POST:
            form = ImageUploadModelForm(request.POST, request.FILES, instance=Image.objects.get(pk=request.POST['image_pk']))
        else:
            form = ImageUploadModelForm(request.POST, request.FILES)
        if form.is_valid():
            # If an image id was in the form, we update the existing image
            form.save()
            return redirect(request.META['HTTP_REFERER'])
        else:
            context = self.get_context_dict(request)
            context.update({"form":form})
            return redirect("composer_modal_image", branding=branding)



class MetadataComposerModalImageLibrary(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_modal_image_library.html"
    
    @method_decorator(login_required)
    @method_decorator(never_cache)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerModalImageLibrary, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        
        context = self.get_context_dict(request)
        context.update({"image_results":get_images(1)})
        return self.render_to_response(context)



