from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.core.paginator import Paginator, InvalidPage, EmptyPage
from django.shortcuts import redirect
from django.utils.decorators import method_decorator
from django.views.generic.base import View, TemplateResponseMixin
from django.views.generic.edit import FormView
from ldt.ldt_utils.models import Project, Content
from metadatacomposer.forms import ImageUploadModelForm
from metadatacomposer.models import Image


import logging #@UnresolvedImport
logger = logging.getLogger(__name__)


class MetadataComposerContextView(View):
    
    branding = "iri"
    
    def get_context_dict(self, request):
        context = {}
        context["branding"] = self.branding
        context["top_header_partial"] = "partial/%s_composer_header.html" % self.branding
        return context



class MetadataComposerHome(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_home.html"
    
    @method_decorator(login_required)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerHome, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        
        projects = Project.safe_objects.filter(owner=request.user).order_by('-modification_date')[:6]
        images = Image.objects.order_by('-modification_date')[:6]
        logger.debug(images)
        contents = Content.safe_objects.order_by('-update_date')[:6]
        
        context = self.get_context_dict(request)
        context.update({"projects":projects, "images": images, "contents": contents})
        return self.render_to_response(context)



class MetadataComposerProjectList(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_project_list.html"
    
    @method_decorator(login_required)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerProjectList, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        page = request.GET.get("page") or 1
        
        projects = Project.safe_objects.prefetch_related("contents").order_by('-modification_date').filter(owner=request.user)
        nb = getattr(settings, 'METADATACOMPOSER_ELEMENTS_PER_PAGE', 9)
        if page=="x":
            nb = projects.count()
        
        paginator = Paginator(projects, nb)
        try:
            results = paginator.page(page)
        except (EmptyPage, InvalidPage):
            results = paginator.page(paginator.num_pages)
        
        context = self.get_context_dict(request)
        context.update({"results":results})
        return self.render_to_response(context)



class MetadataComposerResourceList(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_resource_list.html"
    
    @method_decorator(login_required)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerResourceList, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        
        # We get the first contents page and theirs projects
        content_results = get_contents_and_projects(1, request.user)
        # We get the first page of images
        
        context = self.get_context_dict(request)
        context.update({"content_results":content_results})
        return self.render_to_response(context)



class MetadataComposerContentPagination(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "partial/resource_content_list.html"
    
    @method_decorator(login_required)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerContentPagination, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        page = request.GET.get("page") or 1
        # Get current contents page and theirs projects
        content_results = get_contents_and_projects(page, request.user)
        
        context = self.get_context_dict(request)
        context.update({"content_results":content_results})
        return self.render_to_response(context)



def get_contents_and_projects(page, user):
    
    # We get the current's page contents
    # prefetch_related("project_set") is unfortunately useless because we have to filter the project queryset later
    contents = Content.safe_objects.order_by('-update_date')
    nb = getattr(settings, 'METADATACOMPOSER_ELEMENTS_PER_PAGE', 9)
    if page=="x":
        nb = contents.count()
    paginator = Paginator(contents, nb)
    try:
        results = paginator.page(page)
    except (EmptyPage, InvalidPage):
        results = paginator.page(paginator.num_pages)
    
    # We add the user's projects for each content
    results_object_list = []
    for content in results.object_list:
        # We filter the content's projects with the user's ones
        projects = content.project_set.all().filter(owner=user)
        results_object_list.append({"content":content, "projects":projects})
    
    results.object_list = results_object_list
    return results



class MetadataComposerModalVideo(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_modal_video.html"
    
    @method_decorator(login_required)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerModalVideo, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        context = self.get_context_dict(request)
        return self.render_to_response(context)



class MetadataComposerModalImage(TemplateResponseMixin, MetadataComposerContextView):
    
    def get_template_names(self):
        return "metadatacomposer_modal_image.html"
    
    @method_decorator(login_required)
    def dispatch(self, *args, **kwargs):
        return super(MetadataComposerModalImage, self).dispatch(*args, **kwargs)
    
    def get(self, request, branding="iri", **kwargs):
        self.branding = branding
        
        # Add form
        form = ImageUploadModelForm()
        
        context = self.get_context_dict(request)
        context.update({"form":form})
        return self.render_to_response(context)
    
    def post(self, request, branding="iri", **kwargs):
        self.branding = branding
        
        # Check form
        form = ImageUploadModelForm(request.POST, request.FILES)
        logger.debug("COUCOU 1")
        logger.debug(form)
        logger.debug(form.is_valid())
        if form.is_valid():
            logger.debug("COUCOU 2")
            # If an image id was in the form, we update the existing image
            if 'image_pk' in request.POST:
                form = ImageUploadModelForm(request.POST, request.FILES, instance=Image.objects.get(pk=request.POST['image_pk']))
            form.save()
        else:
            logger.debug("COUCOU 3")
            context = self.get_context_dict(request)
            context.update({"form":form})
        return redirect(request.META['HTTP_REFERER'])



