# -*- coding: utf-8 -*-
'''
Created on Jun 12, 2013

@author: ymh
'''

from .forms import (ModifyWpLinkForm, ValidateTermForm, RemoveWpLinkForm, 
    TermFilterForm)
from .utils import JocondePaginator
from core.models import Term, TERM_URL_STATUS_DICT
from core.models.term import (TERM_WK_LINK_SEMANTIC_LEVEL_DICT, 
    TERM_WK_LINK_SEMANTIC_LEVEL_CHOICES)
from core.wp_utils import process_term as wp_process_term
from django.conf import settings
from django.db.models import Count
from django.http.response import HttpResponse
from django.views.generic import ListView, DetailView, View
from jocondelab.forms import WikipediaEditionForm, LinkSemanticLevelForm
import logging
import urllib

logger = logging.getLogger(__name__)

class TermListView(ListView):
    
    model = Term
    paginate_by = settings.TERM_LIST_PAGE_SIZE
    paginator_class = JocondePaginator
    template_name = "jocondelab/term_list.html"
    filter_form_class = TermFilterForm
    
    def get_filter_form(self):
        initial = { 'order_by':'label',
                    'order_dir': 'asc',
                    'thesaurus': None,
                    'label': None,
                    'link_status': -1,
                    'validated': None}
        return self.filter_form_class(self.request.GET, initial=initial, auto_id=True)
    
    def get_context_data(self, **kwargs):
        context = ListView.get_context_data(self, **kwargs)
        context['filter_form'] = self.get_filter_form()
        return context
    
    def get_queryset(self):
        qs = Term.objects.values('id').annotate(nb_notice=Count("noticeterm"))
        filter_form = self.get_filter_form()
        if filter_form.is_valid():
            return filter_form.get_filter_qs(qs)
        else:
            return None

class TermListTableView(TermListView):
    
    template_name = "jocondelab/partial/term_list_table.html"
    
    

class TermEditView(DetailView):
    
    queryset = Term.objects.select_related()
    pk_url_kwarg = "term_id"
    context_object_name = "term"
    template_name = "jocondelab/term_edit.html"
    filter_form_class = TermFilterForm
    
    def get_filter_form(self):
        initial = { 'order_by':'label',
                    'order_dir': 'asc',
                    'thesaurus': None,
                    'label': None,
                    'link_status': -1,
                    'validated': None}
        return self.filter_form_class(self.request.GET, initial=initial, auto_id=True)

    
    def get_context_data(self, **kwargs):        
        context = DetailView.get_context_data(self, **kwargs)
        context['filter_form'] = self.get_filter_form()
        context['link_semantic_level_choice'] = TERM_WK_LINK_SEMANTIC_LEVEL_CHOICES
        
        field_index = {
            'DOMN' : 1,
            'AUTR' : 3,
            'ECOL' : 4,
            'REPR' : 5,
            'PERI' : 6,
            'EPOQ' : 6,
            'LIEUX': 4,
            'SREP' : 9
        }[self.object.thesaurus.label]
        
        field_name = {
            'SREP' :  u"Source sujet représenté"
        }.get(self.object.thesaurus.label, self.object.thesaurus.label) 

        encoded_label = self.object.label.encode('latin1') if self.object.label is not None else ""
        
        context['encoded_term_label_query_parameter'] = urllib. urlencode({
                'FIELD_%d' % field_index: field_name.encode('latin1'),
                'VALUE_%d' % field_index: encoded_label}).replace('+','%20')
        return context
        
        
    
class TermUpdateView(View):
    
    form_class = None
    http_method_names = ['post']
    
    def __init__(self, **kwargs):
        View.__init__(self, **kwargs)
        self.form = None
        self.form_values = None
        self.term = None
    
    def post(self, request, *args, **kwargs):
        self.form = self.form_class(request.POST)
        if not self.form.is_valid():
            return HttpResponse("Parameters not valid : %s" % (self.form.cleaned_data), status=400)
        
        self.form_values = self.form.cleaned_data
        
        try:
            self.term = Term.objects.get(id=self.form_values['term_id'])
        except Term.DoesNotExist:
            return HttpResponse("Term %d not found" % self.form_values['term_id'],status=404)

        return self.process_term(request)

    def process_term(self, request):
        raise NotImplemented()


class TermValidate(TermUpdateView):
    
    form_class = ValidateTermForm
    
    def process_term(self, request):
        if self.form_values['validation_val']:
            self.term.validate(request.user)            
        else:
            self.term.unvalidate()

        return HttpResponse(status=204)


class TermRemoveWpLink(TermUpdateView):

    form_class = RemoveWpLinkForm
    
    def process_term(self, request):
                        
        self.term.wp_label = None
        self.term.wp_alternative_label = None
        self.term.alternative_wikipedia_url = None
        self.term.alternative_wikipedia_pageid = None
        self.term.wikipedia_url =None
        self.term.wikipedia_pageid = None
        self.term.dbpedia_uri = None
        self.term.wikipedia_revision_id = None
        self.term.url_status = TERM_URL_STATUS_DICT["unsemantized"]
        self.link_semantic_level = TERM_WK_LINK_SEMANTIC_LEVEL_DICT['--']
        
        self.term.save()
                
        return HttpResponse(status=204)



class TermModifyWpLink(TermUpdateView):    
    
    form_class = ModifyWpLinkForm
        
    def process_term(self, request):
        
        label = self.form_values['label']
                
        wp_process_term(None, self.term, label)
        
        return HttpResponse(status=204)


class TermWikipediaEdition(TermUpdateView):    
    
    form_class = WikipediaEditionForm
        
    def process_term(self, request):
        
        self.term.wikipedia_edition = self.form_values['wikipedia_edition']
        self.term.save()
        
        return HttpResponse(status=204)
 
class TermLinkSemanticLevelEdition(TermUpdateView):

    form_class = LinkSemanticLevelForm
     
    def process_term(self, request):
        
        self.term.link_semantic_level = self.form_values['link_semantic_level']
        self.term.save()
        
        return HttpResponse(status=204)
        