# -*- coding: utf-8 -*-
'''
Created on Jun 8, 2013

@author: ymh
'''

from .. import settings
from django.contrib.auth import get_user_model
from django.db import models
from django.utils.translation import ugettext as _
from mptt.models import MPTTModel, TreeForeignKey
import datetime
import logging

logger = logging.getLogger(__name__)

User = get_user_model()

TERM_URL_STATUS_CHOICES = (
    (0, "null_result"),
    (1, "redirection"),
    (2, "homonyme"),
    (3, "match"),
    (4, "unsematized"),
)

TERM_URL_STATUS_CHOICES_TRANS = (
    (0, _("null_result")),
    (1, _("redirection")),
    (2, _("homonyme")),
    (3, _("match")),
    (4, _("unsematized")),
)

TERM_URL_STATUS_DICT = {
    "null_result":0,
    "redirection":1,
    "homonyme":2,
    "match":3,
    "unsemantized":4,
}

TERM_WK_LINK_SEMANTIC_LEVEL_CHOICES = (
    (0, "--"),
    (1, "EE"),
    (2, "EI"),
    (3, "BM"),
    (4, "NM")
)

TERM_WK_LINK_SEMANTIC_LEVEL_CHOICES_TRANS = (
    (0, _("--")),
    (1, _("EE")),
    (2, _("EI")),
    (3, _("BM")),
    (4, _("NM"))
)

TERM_WK_LINK_SEMANTIC_LEVEL_DICT = {
    "--" : 0,
    "EE" : 1,
    "EI" : 2,
    "BM" : 3,
    "NM" : 4
}

   
class Thesaurus(models.Model):
    label = models.CharField(max_length=128, unique=True, blank=False, null=False, db_index=True)
    title = models.CharField(max_length=1024, unique=False, blank=False, null=False, db_index=False)
    description = models.TextField(blank=True, null=True)    
    uri = models.URLField(max_length=2048, blank=True, null=True, db_index=True)
    
    class Meta:
        app_label = 'core'
        ordering = ['label']
        
    def __unicode__(self):
        return self.label


class Term(MPTTModel):
    label = models.CharField(max_length=1024, unique=False, blank=False, null=False, db_index=True)
    lang = models.CharField(max_length=128, unique=False, blank=True, null=True, db_index=True)
    uri = models.URLField(max_length=2048, blank=True, null=True, db_index=True)
    normalized_label = models.CharField(max_length=1024, unique=False, blank=False, null=False, db_index=True, editable=False)
    wp_label = models.CharField(max_length=1024, unique=False, blank=True, null=True, db_index=True)
    wp_alternative_label = models.CharField(max_length=1024, unique=False, blank=True, null=True, db_index=True)
    thesaurus = models.ForeignKey(Thesaurus, blank=False, null=False, db_index=True)
    created_at = models.DateTimeField(auto_now_add=True)
    wikipedia_url = models.URLField(max_length=2048, blank=True, null=True, db_index=True)
    wikipedia_pageid = models.BigIntegerField(unique=False, blank=True, null=True, db_index=True)
    wikipedia_revision_id = models.BigIntegerField(unique=False, blank=True, null=True)
    alternative_wikipedia_url = models.URLField(max_length=2048, blank=True, null=True, db_index=True)
    alternative_wikipedia_pageid = models.BigIntegerField(unique=False, blank=True, null=True, db_index=True)
    url_status = models.IntegerField(choices=TERM_URL_STATUS_CHOICES_TRANS, blank=True, null=True, default=None, db_index=True)
    link_semantic_level = models.IntegerField(choices=TERM_WK_LINK_SEMANTIC_LEVEL_CHOICES_TRANS, blank=True, null=True, default=None, db_index=True)
    dbpedia_uri = models.URLField(max_length=2048, blank=True, null=True, db_index=True)
    validation_date = models.DateTimeField(null=True, blank=True, serialize=False)
    validated = models.BooleanField(default=False, db_index=True)
    validator = models.ForeignKey(User, null=True, blank=True, serialize=False)
    wikipedia_edition = models.BooleanField(default=False, blank=False, null=False)
    
    nb_notice = models.IntegerField(blank=False, null=False, default=0, db_index=True, editable=False)
    notices = models.ManyToManyField('core.Notice', related_name="terms+", through="core.NoticeTerm")
    
    parent = TreeForeignKey('self', null=True, blank=True, related_name='children')
    
    @property
    def alternative_labels_str(self):
        return " | ".join([l.label for l in self.alternative_labels.all() if l.label != self.label])
    
    @property
    def alternative_labels_list(self):
        return [l.label for l in self.alternative_labels.all() if l.label != self.label]
    
    @property
    def wikipedia_revision_permalink(self):
        return settings.WIKIPEDIA_VERSION_PERMALINK_TEMPLATE % (unicode(self.wikipedia_revision_id))
    
    @property
    def url_status_text(self):
        return TERM_URL_STATUS_CHOICES[self.url_status][1]

    @property
    def url_status_text_trans(self):
        return TERM_URL_STATUS_CHOICES_TRANS[self.url_status][1]
        
    def validate(self, user):
        if not self.validated:
            self.validation_date = datetime.datetime.utcnow()
            self.validated = True
            self.validator = user
            self.save()
    
    def unvalidate(self):
        if self.validated:
            self.validated = False
            self.validator = None
            self.validation_date = None
            self.save()
    
    class Meta:
        app_label = 'core'
        
    class MPTTMeta:
        order_insertion_by = ['normalized_label']

class TermLabel(models.Model):
    label = models.CharField(max_length=1024, unique=False, blank=False, null=False, db_index=True)
    lang = models.CharField(max_length=128, unique=False, blank=True, null=True, db_index=True)
    term = models.ForeignKey(Term, blank=False, null=False, db_index=True, related_name="alternative_labels")
    
    class Meta:
        app_label = 'core'
