# -*- coding: utf-8 -*-
'''
Created on Aug 08, 2013

@author: rvelt
'''

from core.models import Notice
from jocondelab.models import DbpediaResource, DbpediaTranslation
from django.conf import settings
from django.views.generic import View
from django.views.generic.base import TemplateResponseMixin
from django.db.models import Q, Count
import random
import re
import time

class MultilingualSearch(View, TemplateResponseMixin):
    
    template_name = "jocondelab/front_search.html"
    
    def get(self, request):
        
        starttime = time.time()
        
        context = {}
        lang = request.GET.get('lang',request.LANGUAGE_CODE)
        querystr = request.GET.get('q', "")
        search_in_title = request.GET.get('search_in_title', True)
        search_in_translations = request.GET.get('search_in_translations', True)
        show_tagcloud = request.GET.get('show_tagcloud', True)
        queryterms = [s.strip(" ") for s in re.split("[,;]",querystr) if s.strip(" ")] if (search_in_title or search_in_translations) else None
        
        npp = 48 if queryterms else 24
        notices = []
        
        if queryterms and search_in_translations and search_in_title:
            mainq = Q()
            if search_in_translations:
                uriq = Q()
                for term in queryterms:
                    uriq = uriq | Q(translations__label__icontains=term)
                ds = DbpediaResource.objects.filter(translations__lang=lang).filter(uriq)
                mainq = mainq | Q(dbpedia_resources__dbpediaresource__in=ds)
            if search_in_title:
                for term in queryterms:
                    mainq = mainq | Q(titr__icontains=term)
            qs = Notice.objects.filter(Q(image=True) & mainq).distinct()
            nbnotices = qs.count()
            ns = qs[:npp]
        else:
            qs = Notice.objects.filter(image=True)
            nbnotices = qs.count()
            ns = qs.order_by('?')[:npp] # --- A bit slow
            # minmax = Notice.objects.aggregate(Min('id'), Max('id'))
            # ns = ns.filter(id__in = [random.randint(minmax['id__min'],minmax['id__max']) for i in range(npp * 3)])[:npp] # --- slightly better
        
        for n in ns:
            terms = [{
                "locale_label": ts.dbpediaresource.translations.get(lang=lang).label,
                "thesaurus": ts.thesaurus.label,
                "dbpedia_uri": ts.dbpediaresource.uri
            } for ts in n.dbpedia_resources.filter(dbpediaresource__translations__lang=lang)]
            termsbythesaurus = {}
            for term in terms:
                if not term["thesaurus"] in termsbythesaurus:
                    termsbythesaurus[term["thesaurus"]] = []
                termsbythesaurus[term["thesaurus"]].append(term)
            noticedict = {
                "id": n.id,
                "imagetitle": n.titr if n.titr else n.deno,
                "title": n.titr,
                "denomination": n.deno,
                "image": n.images.all()[0].url if n.images.count() else "",
                "author": n.autr,
                "authors": re.split("\s?;\s?", n.autr) if n.autr else [],
                "all_terms": terms,
                "terms_by_thesaurus": termsbythesaurus
            }
            notices.append(noticedict)
        
        context["lang"] = lang
        context["rescount"] = nbnotices
        context["notices"] = notices
        context["queryterms"] = querystr
        
        # The word cloud is very time-consuming !
        
        wpp = 30
        if show_tagcloud:
            wqs = DbpediaResource.objects.filter(translations__lang=lang, notices__notice__in=qs).annotate(notice_count=Count('notices')).order_by("-notice_count")[:wpp]
         
            words = [{
                "uri": w.uri,
                "label": w.translations.get(lang=lang).label,
                "notice_count": w.notice_count
            } for w in wqs]
        else:
            words = []
        if words:
            fontmax = 2.5
            fontmin = 1.
            scale = (fontmax - fontmin)/max(1,words[0]["notice_count"]-words[-1]["notice_count"])
             
            for w in words:
                w["font_size"] = fontmin + scale * w["notice_count"]
         
        context["words"] = words
        
        context['JOCONDE_IMG_BASE_URL'] = settings.JOCONDE_IMG_BASE_URL
        
        context['duration'] = "%.2f"%(time.time() - starttime)
        
        return self.render_to_response(context)

class NoticeView(View, TemplateResponseMixin):
    
    template_name = "jocondelab/front_notice.html"
    
    def get(self, request, notice_id):
        
        context = {}
        lang = request.GET.get('lang',request.LANGUAGE_CODE)
        
        notice = Notice.objects.get(id=notice_id)
        
        context["notice"] = notice
        context["title"] = notice.titr if notice.titr else notice.deno
        context["images"] = [i.url for i in notice.images.exclude(relative_url__endswith='v.jpg')]
        terms = [{
                  "label": nt.term.label,
                  "thesaurus": nt.term.thesaurus.label,
                  "dbpedia_uri": nt.term.dbpedia_uri
                  } for nt in notice.noticeterm_set.select_related('term__thesaurus').all()]
        uris = [t["dbpedia_uri"] for t in terms if t["dbpedia_uri"]]
        lls = DbpediaTranslation.objects.filter(lang=lang, dbpediaresource__uri__in=uris).all()
        locale_labels = {l.dbpediaresource.uri: l.label for l in lls}
        for t in terms:
            t["translated"] = (t["dbpedia_uri"] in locale_labels)
            t["locale_label"] = locale_labels.get(t["dbpedia_uri"], t["label"])
        termsbythesaurus = {}
        for term in terms:
            if not term["thesaurus"] in termsbythesaurus:
                termsbythesaurus[term["thesaurus"]] = {
                   "translated": [],
                   "untranslated": []
                   }
            termsbythesaurus[term["thesaurus"]]["translated" if term["translated"] else "untranslated"].append(term)
        
        context["terms_by_thesaurus"] = termsbythesaurus
        
        context['JOCONDE_IMG_BASE_URL'] = settings.JOCONDE_IMG_BASE_URL
        
        return self.render_to_response(context)