import { sessions } from '../sessionsReducer';
import SessionRecord from '../../store/sessionRecord';
import * as types from '../../constants/actionTypes';
import { ActionEnum } from '../../constants';


describe('sessions reducer lastSync', () => {

  it('should return the initial state', () => {
    expect(
      sessions(undefined, {})
    ).toEqual([])
  });

  it('should handle types.CREATE_SESSION'  , () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.CREATE_SESSION,
        session: {
          _id: 'session4',
          title: 'Session title 4'
        }
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' }),
      SessionRecord({ _id: 'session4', title: 'Session title 4' })
    ]);

  });

  it('should handle types.UPDATE_SESSION'  , () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.UPDATE_SESSION,
        sessionId: 'session2',
        values: {
          title: 'New session title 2'
        }
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'New session title 2', action: ActionEnum.UPDATED }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ]);
  });


  it('should handle types.UPDATE_SESSION not found', () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.UPDATE_SESSION,
        sessionId: 'session0',
        values: {
          title: 'New session title 0'
        }
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ]);
  });

  it('should handle types.UPDATE_SESSION CREATED'  , () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2', action: ActionEnum.CREATED }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.UPDATE_SESSION,
        sessionId: 'session2',
        values: {
          title: 'New session title 2'
        }
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'New session title 2', action: ActionEnum.CREATED }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ]);
  });


  it('should handle types.UPDATE_SESSION DELETED'  , () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2', action: ActionEnum.DELETED }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.UPDATE_SESSION,
        sessionId: 'session2',
        values: {
          title: 'New session title 2'
        }
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'New session title 2', action: ActionEnum.DELETED }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ]);
  });


  it('should handle types.DO_DELETE_SESSION', () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.DO_DELETE_SESSION,
        sessionId: 'session2'
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ]);

  });

  it('should handle types.DO_DELETE_SESSION unknown', () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.DO_DELETE_SESSION,
        sessionId: 'session0'
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ]);

  });


  it('should handle types.DELETE_SESSION'  , () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.DELETE_SESSION,
        sessionId: 'session2'
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2', action: ActionEnum.DELETED }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ]);

  });

  it('should handle types.DELETE_SESSION unknown'  , () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.DELETE_SESSION,
        sessionId: 'session0'
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ]);

  });

  it('should handle types.LOAD_SESSIONS'  , () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.LOAD_SESSIONS,
        sessions: [
          SessionRecord({ _id: 'session1bis', title: 'Session title 1 bis' }),
          SessionRecord({ _id: 'session2bis', title: 'Session title 2 bis' }),
          SessionRecord({ _id: 'session3bis', title: 'Session title 3 bis' })
        ]
      })
    ).toEqual([
      SessionRecord({ _id: 'session1bis', title: 'Session title 1 bis' }),
      SessionRecord({ _id: 'session2bis', title: 'Session title 2 bis' }),
      SessionRecord({ _id: 'session3bis', title: 'Session title 3 bis' })
    ]);

  });

  it('should handle types.LOAD_SESSION', () => {
    const newSession = SessionRecord({ _id: 'session2', title: 'New session title 2' });
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.LOAD_SESSION,
        session: newSession,
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'New session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ]);

  });

  it('should handle types.LOAD_SESSION new', () => {
    const newSession = SessionRecord({ _id: 'session0', title: 'Session title 0' });
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' })
    ];

    expect(
      sessions(initialState, {
        type: types.LOAD_SESSION,
        session: newSession,
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1' }),
      SessionRecord({ _id: 'session2', title: 'Session title 2' }),
      SessionRecord({ _id: 'session3', title: 'Session title 3' }),
      SessionRecord({ _id: 'session0', title: 'Session title 0' })
    ]);

  });


  it('should handle types.SYNC_RESET_ALL'  , () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1', action: ActionEnum.DELETED }),
      SessionRecord({ _id: 'session2', title: 'Session title 2', action: ActionEnum.CREATED }),
      SessionRecord({ _id: 'session3', title: 'Session title 3', action: ActionEnum.UPDATED }),
      SessionRecord({ _id: 'session4', title: 'Session title 4', action: ActionEnum.NONE })
    ];

    expect(
      sessions(initialState, {
        type: types.SYNC_RESET_ALL
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1', action: ActionEnum.NONE }),
      SessionRecord({ _id: 'session2', title: 'Session title 2', action: ActionEnum.NONE }),
      SessionRecord({ _id: 'session3', title: 'Session title 3', action: ActionEnum.NONE }),
      SessionRecord({ _id: 'session4', title: 'Session title 4', action: ActionEnum.NONE })
    ]);

  });

  it('should handle types.RESET_ACTION_SESSION', () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1', action: ActionEnum.DELETED }),
      SessionRecord({ _id: 'session2', title: 'Session title 2', action: ActionEnum.CREATED }),
      SessionRecord({ _id: 'session3', title: 'Session title 3', action: ActionEnum.UPDATED }),
      SessionRecord({ _id: 'session4', title: 'Session title 4', action: ActionEnum.NONE })
    ];

    expect(
      sessions(initialState, {
        type: types.RESET_ACTION_SESSION,
        sessionId: 'session2'
      })
    ).toEqual([
      SessionRecord({ _id: 'session1', title: 'Session title 1', action: ActionEnum.DELETED }),
      SessionRecord({ _id: 'session2', title: 'Session title 2', action: ActionEnum.NONE }),
      SessionRecord({ _id: 'session3', title: 'Session title 3', action: ActionEnum.UPDATED }),
      SessionRecord({ _id: 'session4', title: 'Session title 4', action: ActionEnum.NONE })
    ]);

  });

  it('should handle types.AUTH_LOGOUT', () => {
    const initialState = [
      SessionRecord({ _id: 'session1', title: 'Session title 1', action: ActionEnum.DELETED }),
      SessionRecord({ _id: 'session2', title: 'Session title 2', action: ActionEnum.CREATED }),
      SessionRecord({ _id: 'session3', title: 'Session title 3', action: ActionEnum.UPDATED }),
      SessionRecord({ _id: 'session4', title: 'Session title 4', action: ActionEnum.NONE })
    ];

    expect(
      sessions(initialState, {
        type: types.AUTH_LOGOUT
      })
    ).toEqual([]);

  });


});
