import logging

from django.contrib.auth import get_user_model
from django.test import TestCase
from protocols.models import (Metacategory, MetacategoryRevision, Protocol,
                              ProtocolRevision)
from protocols.serializers import ProtocolRevisionSerializer

logger = logging.getLogger(__name__)

class ManagerTest(TestCase):

    def setUp(self):
        User = get_user_model()
        self.user1 = User.objects.create_user(
            username='user1',
            email='user1@email.com',
            password='hiddenpassword'
        )

        self.user2 = User.objects.create_user(
            username='user2',
            email='user2@email.com',
            password='hiddenpassword'
        )


    def test_create_new_protocol_new_categories(self):

        protocol_nb = Protocol.objects.all().count()
        protocol_rev_nb = ProtocolRevision.objects.all().count()
        metacategory_nb = Metacategory.objects.all().count()
        metacategory_rev_nb = MetacategoryRevision.objects.all().count()

        revision_def = {
            'title': 'protocol1',
            'description': 'Protocol nº1',
            'owner': 'admin',
            'metacategories' : [{
                'title': 'Important',
                'label': 'important',
                'description': "Important.",
                'color': '#F1C40F',
                'version': 1,
                'has_comment': False
            }, {
                'title': 'Mot-clé',
                'label': 'mot-cle',
                'description': "Mot-clé.",
                'color': '#2ECC71',
                'version': 1,
                'has_comment': False
            }, {
                'title': 'Commentaire',
                'label': 'commentaire',
                'description': "Commentaire.",
                'color': '#3498DB',
                'version': 1,
                'has_comment': True
            }]
        }

        protocol_revision = Protocol.objects.create_new_revision(None, revision_def, None)

        self.assertEqual(protocol_nb+1, Protocol.objects.all().count())
        self.assertEqual(protocol_rev_nb+1, ProtocolRevision.objects.all().count())
        self.assertEqual(metacategory_nb+3, Metacategory.objects.all().count())
        self.assertEqual(metacategory_rev_nb+3, MetacategoryRevision.objects.all().count())


    def test_no_create_new_protocol(self):

        revision_def = {
            'title': 'protocol1',
            'description': 'Protocol nº1',
            'owner': 'admin',
            'metacategories' : [{
                'title': 'Important',
                'label': 'important',
                'description': "Important.",
                'color': '#F1C40F',
                'version': 1,
                'has_comment': False
            }, {
                'title': 'Mot-clé',
                'label': 'mot-cle',
                'description': "Mot-clé.",
                'color': '#2ECC71',
                'version': 1,
                'has_comment': False
            }, {
                'title': 'Commentaire',
                'label': 'commentaire',
                'description': "Commentaire.",
                'color': '#3498DB',
                'version': 1,
                'has_comment': True
            }]
        }

        protocol_revision = Protocol.objects.create_new_revision(None, revision_def, None)

        serializer = ProtocolRevisionSerializer(protocol_revision)

        protocol_nb = Protocol.objects.all().count()
        protocol_rev_nb = ProtocolRevision.objects.all().count()
        metacategory_nb = Metacategory.objects.all().count()
        metacategory_rev_nb = MetacategoryRevision.objects.all().count()

        protocol_revision = Protocol.objects.create_new_revision(protocol_revision.protocol.ext_id, serializer.data, None)

        self.assertEqual(protocol_nb, Protocol.objects.all().count())
        self.assertEqual(metacategory_nb, Metacategory.objects.all().count())
        self.assertEqual(metacategory_rev_nb, MetacategoryRevision.objects.all().count())

        self.assertEqual(protocol_rev_nb, ProtocolRevision.objects.all().count())

    def test_create_new_protocol_title(self):

        revision_def = {
            'title': 'protocol1',
            'description': 'Protocol nº1',
            'owner': 'admin',
            'metacategories' : [{
                'title': 'Important',
                'label': 'important',
                'description': "Important.",
                'color': '#F1C40F',
                'version': 1,
                'has_comment': False
            }, {
                'title': 'Mot-clé',
                'label': 'mot-cle',
                'description': "Mot-clé.",
                'color': '#2ECC71',
                'version': 1,
                'has_comment': False
            }, {
                'title': 'Commentaire',
                'label': 'commentaire',
                'description': "Commentaire.",
                'color': '#3498DB',
                'version': 1,
                'has_comment': True
            }]
        }

        protocol_revision = Protocol.objects.create_new_revision(None, revision_def, None)

        serializer = ProtocolRevisionSerializer(protocol_revision)

        protocol_nb = Protocol.objects.all().count()
        protocol_rev_nb = ProtocolRevision.objects.all().count()
        metacategory_nb = Metacategory.objects.all().count()
        metacategory_rev_nb = MetacategoryRevision.objects.all().count()

        new_data = dict(serializer.data)
        new_data['title'] = 'protocol1-bis'
        protocol_revision = Protocol.objects.create_new_revision(protocol_revision.protocol.ext_id, new_data, None)

        self.assertEqual(protocol_nb, Protocol.objects.all().count())
        self.assertEqual(metacategory_nb, Metacategory.objects.all().count())
        self.assertEqual(metacategory_rev_nb, MetacategoryRevision.objects.all().count())

        self.assertEqual(protocol_rev_nb+1, ProtocolRevision.objects.all().count())


    def test_create_new_protocol_revision_new_categories(self):

        revision_def = {
            'title': 'protocol1',
            'description': 'Protocol nº1',
            'owner': 'admin',
            'metacategories' : [{
                'title': 'Important',
                'label': 'important',
                'description': "Important.",
                'color': '#F1C40F',
                'version': 1,
                'has_comment': False
            }, {
                'title': 'Mot-clé',
                'label': 'Mot-cle',
                'description': "Mot-clé.",
                'color': '#2ECC71',
                'version': 1,
                'has_comment': False
            }, {
                'title': 'Commentaire',
                'label': 'commentaire',
                'description': "Commentaire.",
                'color': '#3498DB',
                'version': 1,
                'has_comment': True
            }]
        }

        protocol_revision = Protocol.objects.create_new_revision(None, revision_def, None)

        protocol_nb = Protocol.objects.all().count()
        protocol_rev_nb = ProtocolRevision.objects.all().count()
        metacategory_nb = Metacategory.objects.all().count()
        metacategory_rev_nb = MetacategoryRevision.objects.all().count()

        revision_def = {
            'version': protocol_revision.version,
            'title': 'protocol1',
            'description': 'Protocol nº1 bis',
            'owner': 'admin',
            'metacategories' : [{
                'title': 'Important',
                'label': 'important',
                'description': "Important.",
                'color': '#F1C40F',
                'version': 1,
                'has_comment': False
            }, {
                'title': 'Mot-clé',
                'label': 'Mot-cle',
                'description': "Mot-clé.",
                'color': '#2ECC71',
                'version': 1,
                'has_comment': False
            }, {
                'title': 'Commentaire',
                'label': 'commentaire',
                'description': "Commentaire.",
                'color': '#3498DB',
                'version': 1,
                'has_comment': True
            }]
        }

        protocol_revision = Protocol.objects.create_new_revision(protocol_revision.protocol.ext_id, revision_def, None)

        self.assertEqual(protocol_nb, Protocol.objects.all().count())
        self.assertEqual(protocol_rev_nb+1, ProtocolRevision.objects.all().count())
        self.assertEqual(metacategory_nb+3, Metacategory.objects.all().count())
        self.assertEqual(metacategory_rev_nb+3, MetacategoryRevision.objects.all().count())

        self.assertEqual(protocol_revision.description, 'Protocol nº1 bis')


    def test_create_new_protocol_revision_no_categories_revisions(self):

        metacategory1 = Metacategory.objects.create(
            app=None,
            title="Important base",
            label="important-base",
            description="Important base.",
            color="#F1C41F",
            has_comment=False,
            is_default=False
        )
        metacategory_revision1 = MetacategoryRevision.objects.create(
            base=metacategory1,
            title="Important",
            label="important",
            description="Important.",
            color="#F1C40F",
            has_comment=False
        )

        metacategory2 = Metacategory.objects.create(
            app=None,
            title="Mot-clé base",
            label="Mot-cle-base",
            description="Mot-clé base.",
            color="#2ECC72",
            has_comment=False,
            is_default=False
        )
        metacategory_revision2 = MetacategoryRevision.objects.create(
            base=metacategory2,
            title="Mot-clé",
            label="mot-cle",
            description="Mot-clé.",
            color="#2ECC71",
            has_comment=False
        )

        metacategory3 = Metacategory.objects.create(
            app=None,
            title="Commentaire base",
            label="commentaire-base",
            description="Commentaire base.",
            color="#3498DC",
            has_comment=False,
            is_default=False
        )
        metacategory_revision3 = MetacategoryRevision.objects.create(
            base=metacategory3,
            title="Commentaire",
            label="commentaire",
            description="Commentaire.",
            color="#3498DB",
            has_comment=True
        )

        revision_def = {
            'title': 'protocol1',
            'description': 'Protocol nº1',
            'owner': 'admin',
            'metacategories' : [{
                'id': str(metacategory_revision1.ext_id),
                'base': str(metacategory1.ext_id),
                'title': 'Important',
                'label': 'important',
                'description': "Important.",
                'color': '#F1C40F',
                'version': 1,
                'has_comment': False
            }, {
                'id': str(metacategory_revision2.ext_id),
                'base': str(metacategory2.ext_id),
                'title': 'Mot-clé',
                'label': 'mot-cle',
                'description': "Mot-clé.",
                'color': '#2ECC71',
                'version': 1,
                'has_comment': False
            }, {
                'id': str(metacategory_revision3.ext_id),
                'base': str(metacategory3.ext_id),
                'title': 'Commentaire',
                'label': 'commentaire',
                'description': "Commentaire.",
                'color': '#3498DB',
                'version': 1,
                'has_comment': True
            }]
        }

        protocol_nb = Protocol.objects.all().count()
        protocol_rev_nb = ProtocolRevision.objects.all().count()
        metacategory_nb = Metacategory.objects.all().count()
        metacategory_rev_nb = MetacategoryRevision.objects.all().count()

        protocol_revision = Protocol.objects.create_new_revision(None, revision_def, None)

        self.assertEqual(protocol_nb+1, Protocol.objects.all().count())
        self.assertEqual(protocol_rev_nb+1, ProtocolRevision.objects.all().count())
        self.assertEqual(metacategory_nb, Metacategory.objects.all().count())
        self.assertEqual(metacategory_rev_nb, MetacategoryRevision.objects.all().count())



    def test_create_new_protocol_revision_one_categories_revisions(self):

        metacategory1 = Metacategory.objects.create(
            app=None,
            title="Important base",
            label="important-base",
            description="Important base.",
            color="#F1C41F",
            has_comment=False,
            is_default=False
        )
        metacategory_revision1 = MetacategoryRevision.objects.create(
            base=metacategory1,
            title="Important",
            label="important",
            description="Important.",
            color="#F1C40F",
            has_comment=False
        )

        metacategory2 = Metacategory.objects.create(
            app=None,
            title="Mot-clé base",
            label="mot-cle-base",
            description="Mot-clé base.",
            color="#2ECC72",
            has_comment=False,
            is_default=False
        )
        metacategory_revision2 = MetacategoryRevision.objects.create(
            base=metacategory2,
            title="Mot-clé",
            label="mot-cle",
            description="Mot-clé.",
            color="#2ECC71",
            has_comment=False
        )

        metacategory3 = Metacategory.objects.create(
            app=None,
            title="Commentaire base",
            label="commentaire-base",
            description="Commentaire base.",
            color="#3498DC",
            has_comment=False,
            is_default=False
        )
        metacategory_revision3 = MetacategoryRevision.objects.create(
            base=metacategory3,
            title="Commentaire",
            label="commentaire",
            description="Commentaire.",
            color="#3498DB",
            has_comment=True
        )

        revision_def = {
            'title': 'protocol1',
            'description': 'Protocol nº1',
            'owner': 'admin',
            'metacategories' : [{
                'id': str(metacategory_revision1.ext_id),
                'base': str(metacategory1.ext_id),
                'title': 'Important',
                'label': 'important',
                'description': "Important.",
                'color': '#F1C40F',
                'version': 1,
                'has_comment': False
            }, {
                'id': str(metacategory_revision2.ext_id),
                'base': str(metacategory2.ext_id),
                'title': 'Mot-clé',
                'label': 'mot-cle',
                'description': "Mot-clé.",
                'color': '#2ECC71',
                'version': 1,
                'has_comment': False
            }, {
                'id': str(metacategory_revision3.ext_id),
                'base': str(metacategory3.ext_id),
                'title': 'Commentaire amélioré',
                'label': 'commentaire-ameliore',
                'description': "Commentaire.",
                'color': '#3498DB',
                'version': 1,
                'has_comment': True
            }]
        }

        protocol_nb = Protocol.objects.all().count()
        protocol_rev_nb = ProtocolRevision.objects.all().count()
        metacategory_nb = Metacategory.objects.all().count()
        metacategory_rev_nb = MetacategoryRevision.objects.all().count()

        protocol_revision = Protocol.objects.create_new_revision(None, revision_def, None)

        self.assertEqual(protocol_nb+1, Protocol.objects.all().count())
        self.assertEqual(protocol_rev_nb+1, ProtocolRevision.objects.all().count())
        self.assertEqual(metacategory_nb, Metacategory.objects.all().count())
        self.assertEqual(metacategory_rev_nb+1, MetacategoryRevision.objects.all().count())

    def test_create_with_revision(self):
        metacategory1 = Metacategory.objects.create_with_revision(
            app=self.user1,
            title="Important base",
            label="important-base",
            description="Important base.",
            color="#F1C41F",
            has_comment=False,
            is_default=False
        )

        self.assertEqual(1, metacategory1.revisions.count())
        self.assertEqual(5, MetacategoryRevision.objects.all().count()) # There are the 4 created by default
        revision = metacategory1.revisions.all()[0]
        self.assertEqual(metacategory1.title, revision.title)
        self.assertEqual(metacategory1.description, revision.description)
        self.assertEqual(metacategory1.color, revision.color)
        self.assertEqual(metacategory1.has_comment, revision.has_comment)


    def test_create_new_protocol_default(self):
        metacategory1 = Metacategory.objects.create_with_revision(
            app=None,
            title="Important base",
            label="important_base",
            description="Important base.",
            color="#F1C41F",
            has_comment=False,
            is_default=True
        )

        metacategory2 = Metacategory.objects.create_with_revision(
            app=None,
            title="Mot-clé base",
            label="mot-clef-base",
            description="Mot-clé base.",
            color="#2ECC72",
            has_comment=False,
            is_default=True
        )

        metacategory3 = Metacategory.objects.create_with_revision(
            app=None,
            title="Commentaire base",
            label="commentaire-base",
            description="Commentaire base.",
            color="#3498DC",
            has_comment=False,
            is_default=True
        )

        metacategory4 = Metacategory.objects.create_with_revision(
            app=None,
            title="Commentaire base",
            label="commentaire-base",
            description="Commentaire base.",
            color="#3498DC",
            has_comment=False,
            is_default=False
        )

        protocol = Protocol.objects.create_from_default('Default', 'Default protocol', 'admin', None)

        self.assertEqual(3, Protocol.objects.count()) # 1 protocol for each user (group) + 1
        self.assertEqual(3, ProtocolRevision.objects.count()) # 1 protocol rev for each user (group) + 1

        self.assertEqual(1, protocol.revisions.count())
        protocol_revision = protocol.revisions.all()[0]
        self.assertEqual(7, protocol_revision.metacategories.count()) # There are the 4 created by default
        title_set = set([mc.title for mc in protocol_revision.metacategories.all()])
        self.assertEqual(set(['trouble', 'important', 'commentaire', 'mot-clef', "Mot-clé base","Commentaire base","Important base"]), title_set)


    def test_create_new_protocol_default_revision(self):
        metacategory1 = Metacategory.objects.create_with_revision(
            app=None,
            title="Important base",
            label="important-base",
            description="Important base.",
            color="#F1C41F",
            has_comment=False,
            is_default=True
        )

        metacategory_revision1 = MetacategoryRevision.objects.create(
            base=metacategory1,
            title="Important",
            label="important",
            description="Important.",
            color="#F1C40F",
            has_comment=False
        )


        metacategory2 = Metacategory.objects.create_with_revision(
            app=None,
            title="Mot-clé base",
            label="mot-clef-base",
            description="Mot-clé base.",
            color="#2ECC72",
            has_comment=False,
            is_default=True
        )

        metacategory_revision2 = MetacategoryRevision.objects.create(
            base=metacategory2,
            title="Mot-clé",
            label="mot-clef",
            description="Mot-clé.",
            color="#2ECC71",
            has_comment=False
        )

        metacategory3 = Metacategory.objects.create_with_revision(
            app=None,
            title="Commentaire base",
            label="commentaire-base",
            description="Commentaire base.",
            color="#3498DC",
            has_comment=False,
            is_default=True
        )

        metacategory_revision3 = MetacategoryRevision.objects.create(
            base=metacategory3,
            title="Commentaire",
            label="commentaire",
            description="Commentaire.",
            color="#3498DB",
            has_comment=True
        )


        protocol = Protocol.objects.create_from_default('Default', 'Default protocol', 'admin', None)

        self.assertEqual(3, Protocol.objects.count()) # 1 protocol for each user (group) + 1
        self.assertEqual(3, ProtocolRevision.objects.count()) # 1 protocol rev for each user (group) + 1

        self.assertEqual(1, protocol.revisions.count())
        protocol_revision = protocol.revisions.all()[0]
        self.assertEqual(7, protocol_revision.metacategories.count()) # There are the 4 created by default
        title_set = set([mc.title for mc in protocol_revision.metacategories.all()])
        self.assertEqual(set(['trouble', 'important', 'commentaire', 'mot-clef', 'Mot-clé base', 'Commentaire base', 'Important base']), title_set)


    def test_create_new_protocol_default_user(self):
        metacategory1 = Metacategory.objects.create_with_revision(
            app=None,
            title="Important base",
            label="important-base",
            description="Important base.",
            color="#F1C41F",
            has_comment=False,
            is_default=True
        )

        metacategory2 = Metacategory.objects.create_with_revision(
            app=None,
            title="Mot-clé base",
            label="mot-clef-base",
            description="Mot-clé base.",
            color="#2ECC72",
            has_comment=False,
            is_default=True
        )

        metacategory3 = Metacategory.objects.create_with_revision(
            app=None,
            title="Commentaire base",
            label="commentaire-base",
            description="Commentaire base.",
            color="#3498DC",
            has_comment=False,
            is_default=True
        )

        metacategory1bis = Metacategory.objects.create_with_revision(
            app=self.user2,
            title="Important base bis",
            label="important-base-bis",
            description="Important base bis.",
            color="#F1C41F",
            has_comment=False,
            is_default=True
        )

        metacategory2bis = Metacategory.objects.create_with_revision(
            app=self.user2,
            title="Mot-clé base bis",
            label="mot-cle-base-bis",
            description="Mot-clé base bis.",
            color="#2ECC72",
            has_comment=False,
            is_default=True
        )


        protocol = Protocol.objects.create_from_default('Default', 'Default protocol', 'admin', None)

        self.assertEqual(3, Protocol.objects.count()) # 1 protocol for each user (group) + 1
        self.assertEqual(3, ProtocolRevision.objects.count()) # 1 protocol rev for each user (group) + 1

        self.assertEqual(1, protocol.revisions.count())
        protocol_revision = protocol.revisions.all()[0]
        self.assertEqual(7, protocol_revision.metacategories.count()) # There are the 4 created by default
        title_set = set([mc.title for mc in protocol_revision.metacategories.all()])
        self.assertEqual(set(['trouble', 'important', 'commentaire', 'mot-clef', 'Mot-clé base', 'Commentaire base', 'Important base']), title_set)
