"""
Core viewsets
"""
import datetime
import logging

from django.utils import timezone
from django_filters.rest_framework import DjangoFilterBackend
from notes.models import Note, Session
from rest_framework import viewsets

from ..filters import NoteFilterSet, SessionFilterSet
from ..permissions import NotePermission, SessionPermission
from ..serializers.core import (CreateNoteSerializer, CreateSessionSerializer,
                                DetailNoteSerializer, DetailSessionSerializer,
                                ListNoteSerializer, ListSessionSerializer,
                                RootDetailNoteSerializer, UpdateNoteSerializer)

logger = logging.getLogger(__name__)


class SessionViewSet(viewsets.ModelViewSet):
    """
    API endpoint that allow sessions ro be viewed or edited
    """
    serializer_class = ListSessionSerializer
    lookup_field = 'ext_id'

    serializers = {
        'list': ListSessionSerializer,
        'retrieve':  DetailSessionSerializer,
        'create':  CreateSessionSerializer,
    }

    permission_classes = (SessionPermission,)

    filter_backends = (DjangoFilterBackend,)
    filter_class = SessionFilterSet

    def get_serializer_class(self):
        return self.serializers.get(self.action, ListSessionSerializer)

    def get_queryset(self):
        queryset = Session.objects.filter(owner=self.request.user).order_by('created')
        modified_since_str = self.request.query_params.get('modified_since', None)
        if modified_since_str is not None:
            modified_since = datetime.datetime.fromtimestamp(
                float(modified_since_str),
                timezone.utc
            )
            queryset = queryset.filter(updated__gte=modified_since)
        return queryset


class NoteViewSet(viewsets.ModelViewSet):

    serializers = {
        'list': ListNoteSerializer,
        'retrieve': DetailNoteSerializer,
        'create': CreateNoteSerializer,
        'update': UpdateNoteSerializer,
    }
    lookup_field = 'ext_id'

    permission_classes = (NotePermission,)
    pagination_class = None

    def get_serializer_class(self):
        return self.serializers.get(self.action, ListNoteSerializer)

    def get_queryset(self):
        return Note.objects.filter(
            session__ext_id=self.kwargs['session_ext_id'],
            session__owner=self.request.user)


class RootNoteViewSet(viewsets.ReadOnlyModelViewSet):

    lookup_field = 'ext_id'

    permission_classes = (NotePermission,)
    serializer_class = RootDetailNoteSerializer

    filter_backends = (DjangoFilterBackend,)
    filter_class = NoteFilterSet

    def get_queryset(self):
        queryset = Note.objects.filter(session__owner=self.request.user).order_by('created')
        modified_since_str = self.request.query_params.get('modified_since', None)
        if modified_since_str is not None:
            modified_since = datetime.datetime.fromtimestamp(
                float(modified_since_str),
                timezone.utc
            )
            queryset = queryset.filter(updated__gte=modified_since)
        return queryset
