import PropTypes from 'prop-types';
import React, { Component } from 'react';
import { connect } from 'react-redux';
import { withRouter } from 'react-router';
import { bindActionCreators } from 'redux';
// import logo from './logo.svg';
import { Navbar, Nav, NavItem, NavDropdown, MenuItem, Modal, Button } from 'react-bootstrap';
import * as authActions from '../actions/authActions';
import { forceSync } from '../actions/networkActions';
import { ActionEnum } from '../constants';

const LoginNav = ({isAuthenticated, currentUser, history, authActions, onLogout}) => {

  const onClickSettings = (e) => {
    e.preventDefault();
    history.push('/settings');
  }

  const onClickLogin = (e) => {
    e.preventDefault();
    history.push('/login');
  }

  if (isAuthenticated) {
    return (
      <NavDropdown title={ currentUser.get('username') } id="user-dropdown">
        <MenuItem onClick={onClickSettings}>Settings</MenuItem>
        <MenuItem onClick={onLogout}>Logout</MenuItem>
      </NavDropdown>
    );
  }
  return (
    <NavItem onClick={onClickLogin} href="/login">Login</NavItem>
  );
}

const Online = ({ online }) => {
  return (
    <NavItem>
      <span className="material-icons" style={{ color: online ? '#2ECC71' : '#E74C3C' }}>signal_wifi_4_bar</span>
    </NavItem>
  )
}

const SyncButton = ({ onSyncClick, isSynchronizing }) => {
  return (
    <NavItem onClick={onSyncClick}>
      Sync
      {isSynchronizing && <span className="material-icons">&#xE627;</span>}
    </NavItem>
  )
}

class AppNavbar extends Component {

  state = {
    showModal: false
  }

  closeModal = () => {
    this.setState({ showModal: false });
  }

  onClickHome = (e) => {
    e.preventDefault();
    this.props.history.push('/');
  }

  isSynchronized = () => {
    return this.props.isSynchronized;
  }

  onClickLogout = (e) => {
    e.preventDefault();
    const isSynchronized = this.isSynchronized();
    if (isSynchronized) {
      this.logout();
    } else {
      this.setState({ showModal: true })
    }
  }

  confirmLogout = () => {
    const isSynchronized = this.isSynchronized();
    if (!isSynchronized) {
      this.props.authActions.resetAll();
    }
    this.logout();
    this.closeModal();
  }

  logout = () => {
    this.props.authActions.logout();
    this.props.history.push('/');
  }

  onClickSessions = (e) => {
    e.preventDefault();
    this.props.history.push('/sessions');
  }

  onSyncClick = (e) => {
    e.preventDefault();
    this.props.networkActions.forceSync();
  }

  render() {
    return (
      <Navbar fluid inverse fixedTop>
        <Navbar.Header>
          <Navbar.Brand>
            <a onClick={this.onClickHome} href="/">IRI Notes</a>
          </Navbar.Brand>
          <Navbar.Toggle />
        </Navbar.Header>
        <Navbar.Collapse>
          <Nav>
            <NavItem onClick={this.onClickSessions} href="/sessions">Sessions</NavItem>
          </Nav>
          <Nav pullRight>
            <SyncButton onSyncClick={this.onSyncClick} isSynchronizing={this.props.isSynchronizing}/>
            <Online {...this.props} />
            <LoginNav {...this.props} onLogout={this.onClickLogout} />
          </Nav>
        </Navbar.Collapse>
        <Modal show={this.state.showModal} onHide={this.closeModal}>
          <Modal.Body>
            <p className="text-center">
              Some data is not synchronized with server.
              <br />
              If you continue, it will be lost.
            </p>
          </Modal.Body>
          <Modal.Footer>
            <Button bsStyle="primary" onClick={this.confirmLogout}>Confirm</Button>
            <Button onClick={this.closeModal}>Close</Button>
          </Modal.Footer>
        </Modal>
      </Navbar>
    );
  }
}

AppNavbar.propTypes = {
  isAuthenticated: PropTypes.bool.isRequired
};

function mapStateToProps(state, props) {
  return {
    isAuthenticated: state.getIn(['authStatus', 'isAuthenticated']),
    currentUser: state.getIn(['authStatus', 'currentUser']),
    online: state.getIn(['status', 'online']),
    isSynchronizing: state.getIn(['status', 'isSynchronizing']),
    isSynchronized: state.get('notes').every((n) => n.get('action')===ActionEnum.NONE) &&
      state.get('sessions').every((n) => n.get('action')===ActionEnum.NONE)
  };
}

function mapDispatchToProps(dispatch) {
  return {
    authActions: bindActionCreators(authActions, dispatch),
    networkActions: bindActionCreators({ forceSync }, dispatch)
  }
}

export default connect(mapStateToProps, mapDispatchToProps)(withRouter(AppNavbar));
