"""
Serializers for model core classes
"""
import logging

from notes.models import Note, Session
from rest_framework import serializers

logger = logging.getLogger(__name__)


class DetailNoteSerializer(serializers.ModelSerializer):
    class Meta:
        model = Note
        fields = (
            'ext_id', 'version', 'created', 'updated',
            'plain', 'html', 'raw',
            'categorization', 'margin_note', 'tc_start', 'tc_end'
        )
        read_only_fields = ('ext_id', 'version', 'created', 'updated')

class UpdateNoteSerializer(serializers.ModelSerializer):
    class Meta:
        model = Note
        fields = (
            'ext_id', 'version', 'created', 'updated',
            'plain', 'html', 'raw',
            'categorization', 'margin_note', 'tc_start', 'tc_end'
        )
        read_only_fields = ('ext_id', 'version', 'created', 'updated', 'tc_start', 'tc_end')

class CreateNoteSerializer(serializers.ModelSerializer):
    class Meta:
        model = Note
        fields = (
            'ext_id', 'version', 'created', 'updated',
            'plain', 'html', 'raw',
            'categorization', 'margin_note', 'tc_start', 'tc_end'
        )
        read_only_fields = ('version', 'created', 'updated')

    def to_internal_value(self, data):
        super_data = super().to_internal_value(data)
        super_data['session'] = Session.objects.get(
            ext_id=self.context['view'].kwargs['session_ext_id']
        )
        return super_data

class ListNoteSerializer(serializers.ModelSerializer):
    class Meta:
        model = Note
        fields = (
            'ext_id', 'tc_start', 'tc_end'
        )
        read_only_fields = ('ext_id', )

class RootListNoteSerializer(serializers.ModelSerializer):
    session = serializers.SlugRelatedField(read_only=True, slug_field='ext_id')

    class Meta:
        model = Note
        fields = (
            'ext_id', 'tc_start', 'tc_end', 'session'
        )
        read_only_fields = ('ext_id', )

class RootDetailNoteSerializer(serializers.ModelSerializer):
    session = serializers.SlugRelatedField(read_only=True, slug_field='ext_id')

    class Meta:
        model = Note
        fields = (
            'ext_id', 'version', 'created', 'updated',
            'plain', 'html', 'raw',
            'categorization', 'margin_note', 'tc_start', 'tc_end',
            'session'
        )
        read_only_fields = (
            'ext_id', 'version', 'created', 'updated',
            'plain', 'html', 'raw',
            'categorization', 'margin_note', 'tc_start', 'tc_end',
            'session'
        )


class ListSessionSerializer(serializers.ModelSerializer):

    owner = serializers.SlugRelatedField(
        read_only=True, slug_field='username', default=serializers.CurrentUserDefault())

    class Meta:
        model = Session
        fields = (
            'ext_id', 'version', 'date', 'created', 'updated',
            'owner', 'title', 'description', 'protocol'
        )
        read_only_fields = ('ext_id', 'version', 'created', 'updated', 'owner')


class DetailSessionSerializer(serializers.ModelSerializer):

    owner = serializers.SlugRelatedField(read_only=True, slug_field='username')
    notes = DetailNoteSerializer(many=True, read_only=True)

    class Meta:
        model = Session
        fields = (
            'ext_id', 'version', 'date', 'created', 'updated',
            'owner', 'title', 'description', 'protocol',
            'notes'
        )
        read_only_fields = ('ext_id', 'version', 'created', 'updated', 'owner')

class CreateSessionSerializer(serializers.ModelSerializer):

    owner = serializers.SlugRelatedField(
        read_only=True, slug_field='username', default=serializers.CurrentUserDefault())


    class Meta:
        model = Session
        fields = (
            'ext_id', 'version', 'date', 'created', 'updated',
            'owner', 'title', 'description', 'protocol'
        )
        read_only_fields = ('version', 'created', 'updated', 'owner')
