import os
try:
    from setuptools import setup
except ImportError:
    from distutils.core import setup
from distutils.command.install_data import install_data
from distutils.command.install import INSTALL_SCHEMES
import sys


class osx_install_data(install_data):
    """
    On MacOS, the platform-specific lib dir is /System/Library/Framework/Python/.../
    which is wrong. Python 2.5 supplied with MacOS 10.5 has an Apple-specific fix
    for this in distutils.command.install_data#306. It fixes install_lib but not
    install_data, which is why we roll our own install_data class.
    """

    def finalize_options(self):
        """
        By the time finalize_options is called, install.install_lib is set to the
        fixed directory, so we set the installdir to install_lib. The
        install_data class uses ('install_data', 'install_dir') instead.
        """
        self.set_undefined_options('install', ('install_lib', 'install_dir'))
        install_data.finalize_options(self)

def fullsplit(path, result=None):
    """
    Split a pathname into components (the opposite of os.path.join) in a
    platform-neutral way.
    """
    if result is None:
        result = []
    head, tail = os.path.split(path)
    if head == '':
        return [tail] + result
    if head == path:
        return result
    return fullsplit(head, [tail] + result)


def launch_setup(setup_script_name, setup_script_args):
    """
    Start setup
    """
    if sys.platform == "darwin":
        cmdclasses = {'install_data': osx_install_data}
    else:
        cmdclasses = {'install_data': install_data}


    root_dir = os.path.dirname(__file__)
    if root_dir != '':
        os.chdir(root_dir)
    source_dirs = ['irinotes', 'notes']

    version_variables = {}
    try:
        with open(os.path.join(source_dirs[0], "__init__.py")) as f:
            code = compile(f.read(), "__init__.py", 'exec')
            exec(code, version_variables)
    except:
        pass

    version = version_variables['__version__']

    packages, data_files = [], []

    for source_dir in source_dirs:
        for dirpath, dirnames, filenames in os.walk(source_dir):
            # Ignore dirnames that start with '.'
            for i, dirname in enumerate(dirnames):
                if dirname.startswith('.') or dirname.startswith('__pycache__'): del dirnames[i]
            if '__init__.py' in filenames:
                packages.append('.'.join(fullsplit(dirpath)))
            elif filenames:
                data_files.append([dirpath, [os.path.join(dirpath, f) for f in filenames]])


    # Tell distutils to put the data_files in platform-specific installation
    # locations. See here for an explanation:
    # http://groups.google.com/group/comp.lang.python/browse_thread/thread/35ec7b2fed36eaec/2105ee4d9e8042cb
    for scheme in INSTALL_SCHEMES.values():
        scheme['data'] = scheme['purelib']

    # Small hack for working with bdist_wininst.
    # See http://mail.python.org/pipermail/distutils-sig/2004-August/004134.html
    if len(sys.argv) > 1 and sys.argv[1] == 'bdist_wininst':
        for file_info in data_files:
            file_info[0] = '\\PURELIB\\%s' % file_info[0]

    #write MANIFEST.in

    with open("MANIFEST.in", "w") as m:
        m.write("include CHANGES\n")
        m.write("include LICENSE\n")
        m.write("include README\n")
        m.write("include MANIFEST.in\n")
        m.write("include requirements/base.txt\n")
        m.write("include requirements/dev.txt\n")
        m.write("include requirements/prod.txt\n")
        for entry in data_files:
            file_list = entry[1]
            for filename in file_list:
                m.write("include %s\n" % (filename))

    long_description = ''
    with open('README', 'r') as f:
        long_description = f.read()

    setup(
        script_name=setup_script_name,
        script_args=setup_script_args,
        name='irinotes',
        version=version,
        author='IRI',
        author_email='contact@iri.centrepompidou.fr',
        packages=packages,
        data_files=data_files,
        cmdclass=cmdclasses,
        scripts=[],
        url='http://www.iri.centrepompidou.fr/dev/hg/irinotes',
        license='CECILL-C',
        description='projet Irinotes',
        long_description=long_description,
        classifiers=[
            'Development Status :: 4 - Beta',
            'Environment :: Web Environment',
            'Framework :: Django',
            'Intended Audience :: Developers',
            'License :: Ceccil-B',
            'Operating System :: OS Independent',
            'Programming Language :: Python',
            'Topic :: Utilities'
        ],
        install_requires=[
            "Django >= 1.11",
            "python-decouple",
            "Unipath",
            "dj-database-url",
            "six",
            "django-auditlog == 0.4.3dev",
            "django-extensions",
            "djangorestframework >= 3.6",
            "django-rest-auth[with_social]",
            "djangorestframework-jwt",
            "django-guardian >= 1.4",
            "django-colorful",
            "django-cors-headers",
            "django-concurrency",
            "django-filter",
            "drf-nested-routers",
            "markdown"
        ],
        dependency_links=[
            "https://github.com/IRI-Research/django-auditlog/tarball/master#egg=django-auditlog-0.4.3dev"
        ]
    )


if __name__ == "__main__":

    script_name = os.path.basename(sys.argv[0])
    script_args = sys.argv[1:]

    launch_setup(script_name, script_args)
