import logging

from django.contrib.auth import get_user_model
from django.test import TestCase
from protocols.models import (Metacategory, MetacategoryRevision, Protocol,
                              ProtocolRevision)
from protocols.serializers import (MetacategoryRevisionSerializer,
                                   ProtocolRevisionSerializer)

# Create your tests here.

logger = logging.getLogger(__name__)

class SerializerTest(TestCase):

    maxDiff=None

    def setUp(self):
        User = get_user_model()
        self.user1 = User.objects.create_user(
            username='user1',
            email='user1@email.com',
            password='hiddenpassword'
        )

    def test_protocol_simple_serialization(self):
        protocol = Protocol.objects.create(app=self.user1, owner='admin')
        protocol_revision = ProtocolRevision.objects.create(title="protocol1", description="Protocol nº1", protocol=protocol)
        serializer = ProtocolRevisionSerializer(protocol_revision)

        expected_data = {
            'id': str(protocol.ext_id),
            'version': 1,
            'title': 'protocol1',
            'description': 'Protocol nº1',
            'owner' : 'admin',
            'metacategories' : []
        }

        self.assertDictEqual(expected_data, serializer.data)


    def test_protocol_complete_serialization(self):
        protocol = Protocol.objects.create(app=self.user1, owner='admin')
        protocol_revision = ProtocolRevision.objects.create(title="protocol2", description="Protocol nº2", protocol=protocol)

        metacategory1 = Metacategory.objects.create(
            app=self.user1,
            title="Important base",
            label="important-base",
            description="Important base.",
            color="#F1C41F",
            has_comment=False,
            is_default=False
        )
        metacategory_revision1 = MetacategoryRevision.objects.create(
            base=metacategory1,
            title="Important",
            label="important",
            description="Important.",
            color="#F1C40F",
            has_comment=False
        )

        metacategory2 = Metacategory.objects.create(
            app=self.user1,
            title="Mot-clé base",
            label="mot-cle-base",
            description="Mot-clé base.",
            color="#2ECC72",
            has_comment=False,
            is_default=False
        )
        metacategory_revision2 = MetacategoryRevision.objects.create(
            base=metacategory2,
            title="Mot-clé",
            label="mot-cle",
            description="Mot-clé.",
            color="#2ECC71",
            has_comment=False
        )

        metacategory3 = Metacategory.objects.create(
            app=self.user1,
            title="Commentaire base",
            label="commentaire-base",
            description="Commentaire base.",
            color="#3498DC",
            has_comment=False,
            is_default=False
        )
        metacategory_revision3 = MetacategoryRevision.objects.create(
            base=metacategory3,
            title="Commentaire",
            label="commentaire",
            description="Commentaire.",
            color="#3498DB",
            has_comment=True
        )

        protocol_revision.metacategories.add(metacategory_revision1, metacategory_revision2, metacategory_revision3)

        serializer = ProtocolRevisionSerializer(protocol_revision)

        expected_data = {
            'id': str(protocol.ext_id),
            'owner': 'admin',
            'version': 1,
            'title': 'protocol2',
            'description': 'Protocol nº2',
            'metacategories' : [{
                'id': str(metacategory_revision1.ext_id),
                'base': str(metacategory1.ext_id),
                'title': 'Important',
                'label': 'important',
                'description': "Important.",
                'color': '#F1C40F',
                'version': 1,
                'has_comment': False
            }, {
                'id': str(metacategory_revision2.ext_id),
                'base': str(metacategory2.ext_id),
                'title': 'Mot-clé',
                'label': 'mot-cle',
                'description': "Mot-clé.",
                'color': '#2ECC71',
                'version': 1,
                'has_comment': False
            }, {
                'id': str(metacategory_revision3.ext_id),
                'base': str(metacategory3.ext_id),
                'title': 'Commentaire',
                'label': 'commentaire',
                'description': "Commentaire.",
                'color': '#3498DB',
                'version': 1,
                'has_comment': True
            }]
        }

        self.assertDictEqual(expected_data, serializer.data)


    def test_metacategory_simple_serialization(self):
        metacategory = Metacategory.objects.create(
            app=self.user1,
            title="Important",
            label="important",
            description="Important.",
            color="#F1C40F",
            has_comment=False,
            is_default=False
        )
        metacategory_revision = MetacategoryRevision.objects.create(
            base=metacategory,
            title="Important",
            label="important",
            description="Important.",
            color="#F1C40F",
            has_comment=False
        )
        serializer = MetacategoryRevisionSerializer(metacategory_revision)

        expected_data = {
            'id': str(metacategory_revision.ext_id),
            'base': str(metacategory.ext_id),
            'title': 'Important',
            'label': 'important',
            'description': "Important.",
            'color': '#F1C40F',
            'version': 1,
            'has_comment': False
        }

        self.assertDictEqual(expected_data, serializer.data)

