import React, { Component } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import { Grid, Row, Col, ListGroup, ListGroupItem, Button, Modal } from 'react-bootstrap';
import moment from 'moment';
import '../App.css';
import Navbar from './Navbar';
import * as sessionsActions from '../actions/sessionsActions';
import uuidV1 from 'uuid/v1';
import { getActiveSessions } from '../selectors/coreSelectors';
import { getCurrentUser, getGroups, getCurrentGroup } from '../selectors/authSelectors';

class SessionList extends Component {

  state = {
    showModal: false,
    sessionToDelete: null
  }

  createSession = () => {
    const sessionId = uuidV1();
    let groupName = null;
    let protocol = null;
    if(this.props.currentGroup) {
      groupName = this.props.currentGroup.get('name');
      protocol = this.props.currentGroup.get('protocol');
    }
    if(groupName === null) {
      groupName = (this.props.currentUser)?this.props.currentUser.get('default_group'):null;
      if(groupName != null) {
        const group = this.props.groups.find((g) => g.name === groupName);
        if(group) {
          protocol = group.get('protocol');
        }
      }
    }

    this.props.sessionsActions.createSession(sessionId, groupName, protocol);
    this.props.history.push('/sessions/' + sessionId);
  }

  onClickDelete(session, e) {
    e.preventDefault();
    e.stopPropagation();

    this.setState({
      showModal: true,
      sessionToDelete: session
    })
  }

  closeModal = () => {
    this.setState({ showModal: false })
  }

  deleteSession = () => {
    const { sessionToDelete } = this.state;

    this.props.sessionsActions.deleteSession(sessionToDelete);

    this.setState({
      showModal: false,
      sessionToDelete: null
    })
  }

  render() {
    return (
      <div>
        <Navbar history={this.props.history} />
        <Grid fluid>
          <Row>
            <Col md={6} mdOffset={3}>
              <ListGroup>
                {this.props.sessions.map((session) =>
                  <ListGroupItem key={session.get('_id')}>
                    <span onClick={() => this.props.history.push('/sessions/' + session.get('_id'))}>{session.title || 'No title'} {session.get('_id')} {moment(session.get('date')).format('DD/MM/YYYY')}</span>
                    <a className="pull-right" onClick={ this.onClickDelete.bind(this, session) }>
                      <span className="material-icons">delete</span>
                    </a>
                  </ListGroupItem>
                )}
              </ListGroup>
              <Button bsStyle="success" onClick={this.createSession}>Create new session</Button>
            </Col>
          </Row>
        </Grid>

        <Modal show={ this.state.showModal } onHide={ this.closeModal }>
          <Modal.Body>
            Are you sure?
          </Modal.Body>
          <Modal.Footer>
            <Button bsStyle="primary" onClick={ this.deleteSession }>Confirm</Button>
            <Button onClick={ this.closeModal }>Close</Button>
          </Modal.Footer>
        </Modal>

      </div>
    );
  }
}

function mapStateToProps(state, props) {
  return {
    sessions: getActiveSessions(state),
    currentUser: getCurrentUser(state),
    groups: getGroups(state),
    currentGroup: getCurrentGroup(state)
  };
}

function mapDispatchToProps(dispatch) {
  return {
    sessionsActions: bindActionCreators(sessionsActions, dispatch),
  }
}

export default connect(mapStateToProps, mapDispatchToProps)(SessionList);
