import Immutable from 'immutable';
import * as types from '../constants/actionTypes';
import NoteRecord from '../store/noteRecord';
import { ActionEnum } from '../constants';

const findNoteIndex = (notes, id) => {
  return notes.findIndex((note) => note.get('_id') === id);
}

const findNote = (notes, id) => {
  return notes.get(findNoteIndex(notes, id));
}

export default (state = Immutable.List([]), action) => {
  switch (action.type) {
    case types.ADD_NOTE: {
      return state.push(new NoteRecord(action.note));
    }
    case types.DELETE_NOTE: {
      const noteIndex = findNoteIndex(state, action.note.get('_id'));
      const note = findNote(state, action.note.get('_id'));
      return state.set(noteIndex, note.merge({action: ActionEnum.DELETED}));
    }
    case types.DO_DELETE_NOTE: {
      const noteIndex = state.findIndex((note) => note.get('_id') === action.noteId);
      return state.delete(noteIndex);
    }
    case types.UPDATE_NOTE: {
      const index = findNoteIndex(state, action.note.get('_id'));
      const note = findNote(state, action.note.get('_id'));
      let newAction;
      switch (note.get('action')) {
        case ActionEnum.CREATED:
          newAction = ActionEnum.CREATED;
          break;
        case ActionEnum.DELETED: // should not happen, but...
          newAction = ActionEnum.DELETED;
          break;
        default:
          newAction = ActionEnum.UPDATED;
      }

      let newNote = note.merge(action.data, {action: newAction});
      return state.set(index, newNote);
    }
    case types.DELETE_SESSION: {
      const sessionId = action.session.get('_id');
      return state.map((note) => {
        if(sessionId === note.session) {
          return note.merge({action: ActionEnum.DELETED});
        } else {
          return note;
        }
      })
    }
    case types.DO_DELETE_SESSION: {
      return state.filter((note) => action.sessionId !== note.session)
    }
    case types.RESET_ACTION_NOTE: {
      const noteId = action.note.get('_id');
      const index = state.findIndex((note) => note.get('_id') === noteId);
      const note = state.get(index);
      return state.set(index, note.merge({action: ActionEnum.NONE}));
    }
    case types.SYNC_RESET_ALL: {
      return state.map((note) => {
        if(note.action !== ActionEnum.NONE) {
          return note.merge({action: ActionEnum.None});
        } else {
          return note;
        }
      });
    }
    case types.LOAD_NOTE: {
      const noteRec = action.note;
      const noteId = noteRec.get('_id');
      const index = state.findIndex((note) => note.get('_id') === noteId);
      if(index >= 0) {
        return state.set(index, noteRec);
      } else {
        return state.push(noteRec);
      }
    }
    case types.AUTH_LOGOUT: {
      return Immutable.List(); // empty note list on logout
    }
    default:
      return state;
  }
};
