import React, { Component } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import { FormGroup, FormControl, Button, InputGroup, HelpBlock, Glyphicon } from 'react-bootstrap';
import * as authActions from '../actions/authActions';
import * as sessionsActions from '../actions/sessionsActions';
import { getOnline, getCreateGroup } from '../selectors/authSelectors';

class GroupForm extends Component {

  state = {
    createGroup: false,
    groupName: ''
  }

  toggleCreateGroup = (e) => {
    e.preventDefault();
    const { createGroup } = this.state;
    this.setState({ createGroup: !createGroup });
  }

  onClickCreateGroup = (e) => {
    e.preventDefault();
    const groupName = this.state.groupName;
    this.props.sessionsActions.createGroupAndUpdateSession(this.props.session, groupName);
    this.setState({
      createGroup: false,
      groupName: ''
    })
  }

  handleInputChange = (e) => {
    const target = e.target;
    const value = target.value;
    const name = target.name;

    this.setState({
      [name]: value
    });
  }


  render = () => {
    const { createGroup } = this.props;
    const hasErrors = true === createGroup.get('error') && createGroup.get('errorMessages').has('name');

    let errors = [];
    if (hasErrors) {
      const errorMessages = createGroup.get('errorMessages').toArray();
      errors = errorMessages.map((message, key) => {
        return (
          <HelpBlock key={ key }>{ message }</HelpBlock>
        )
      })
    }

    if (this.state.createGroup) {
      return (
        <FormGroup validationState={ hasErrors ? 'error' : null }>
          <InputGroup>
            <FormControl
              type="text"
              placeholder="Enter a name for your group"
              onChange={this.handleInputChange}
              name="groupName"
              value={this.state.groupName} />
            <InputGroup.Button>
              <Button bsStyle="primary" onClick={ this.onClickCreateGroup }>Create</Button>
            </InputGroup.Button>
          </InputGroup>
          { errors }
          <hr />
          <Button onClick={ this.toggleCreateGroup }>Cancel</Button>
        </FormGroup>
      )
    }

    if(this.props.online) {
      return (
        <FormGroup>
          <Button className="pull-right" bsSize="small" onClick={ this.toggleCreateGroup }>
            <Glyphicon glyph="plus" />  Create a new group
          </Button>
        </FormGroup>
      )
    }
    return null;
  }

}

function mapStateToProps(state, props) {

  return {
    createGroup: getCreateGroup(state),
    online: getOnline(state),
  };
}

function mapDispatchToProps(dispatch) {
  return {
    sessionsActions: bindActionCreators(sessionsActions, dispatch),
    authActions: bindActionCreators(authActions, dispatch),
  }
}

export default connect(mapStateToProps, mapDispatchToProps)(GroupForm);
