import React, { Component } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import PropTypes from 'prop-types';
import Immutable from 'immutable';
import { Alert, Modal, Button } from 'react-bootstrap';
import Note from './Note';
import * as notesActions from '../actions/notesActions';

class NotesList extends Component {

  state = {
    editingNote: null,
    noteToDelete: null,
    showModal: false,
  }

  enterEditMode = (note) => {
    this.setState({ editingNote: note });
  }

  closeNote = () => {
    this.setState({ editingNote: null });
  }

  confirmDelete = (note) => {
    this.setState({
      showModal: true,
      noteToDelete: note
    })
  }

  deleteNote = () => {
    const { noteToDelete } = this.state;
    this.props.notesActions.deleteNote(noteToDelete);
    this.closeModal();
  }

  closeModal = () => {
    this.setState({
      showModal: false,
      noteToDelete: null
    })
  }

  updateNote = (note, data) => {
    this.props.notesActions.updateNote(note, data);
  }

  render() {
    if (this.props.notes.size === 0) {
      return (
        <Alert bsStyle="warning">No notes yet. Add notes with the textarea below.</Alert>
      );
    }

    return (
      <div>

        {this.props.notes.map((note) =>
          <Note
            note={ note }
            key={ note._id }
            onClick={ this.enterEditMode.bind(this, note) }
            onClose={ this.closeNote }
            onDelete={ this.confirmDelete.bind(this, note) }
            onSave={ this.updateNote }
            isEditing={ this.state.editingNote && note === this.state.editingNote } />
        )}

        <Modal show={this.state.showModal} onHide={this.closeModal}>
          <Modal.Body>
            Are you sure?
          </Modal.Body>
          <Modal.Footer>
            <Button bsStyle="primary" onClick={ this.deleteNote }>Confirm</Button>
            <Button onClick={ this.closeModal }>Close</Button>
          </Modal.Footer>
        </Modal>

      </div>
    );
  }
};

NotesList.propTypes = {
  notes: PropTypes.instanceOf(Immutable.List).isRequired
};

function mapStateToProps(state, props) {
  return props;
}

function mapDispatchToProps(dispatch) {
  return {
    notesActions: bindActionCreators(notesActions, dispatch),
  }
}

export default connect(mapStateToProps, mapDispatchToProps)(NotesList);
