import { select } from 'redux-saga/effects'
import { getCreatedSessions, getUpdatedSessions, getDeletedSessions } from './selectors';
import { ActionEnum } from '../constants';
import moment from 'moment';
import SessionRecord from '../store/sessionRecord';
import { doDeleteSession, loadSession, resetActionSession } from '../actions/sessionsActions';
import Immutable from 'immutable';
import SyncMixin from './BaseSyncronizer';

class SessionSyncBase {

  // local diffs (immutable)
  * getLocalDiffs() {
    return Immutable.Map({
      created: yield select(getCreatedSessions),
      updated: yield select(getUpdatedSessions),
      deleted: yield select(getDeletedSessions)
    })
  }

  // remote urls
  getRemoteLoadUrl() {
    return "/api/notes/sessions/";
  }

  getRemoteDeleteUrl(localObjInst) {
    return `/api/notes/sessions/${localObjInst.get('_id')}/`;
  }

  getRemoteCreateUrl(localObjInst) {
    return "/api/notes/sessions/";
  }

  getRemoteUpdateUrl(localObjInst) {
    return `/api/notes/sessions/${localObjInst.get('_id')}/`;
  }

  // build remote json message
  getRemoteData(localObjInst) {
    return {
      ext_id: localObjInst.get('_id'),
      date: localObjInst.get('date'),
      title: localObjInst.get('title'),
      description: localObjInst.get('description'),
      protocol: ''
    };
  }

  getLocalRecord(remoteObj) {
    return new SessionRecord({
        _id: remoteObj.ext_id,
        title: remoteObj.title,
        description: remoteObj.description,
        date: moment(remoteObj.date).toDate(),
        action: ActionEnum.NONE,
        group: null
    });
  }

  // actions
  doDeleteLocalObj(localObjId) {
    return doDeleteSession(localObjId);
  }

  resetLocalObj(localObjInst) {
    return resetActionSession(localObjInst);
  }

  loadObj(objRecord) {
    return loadSession(objRecord);
  }

}

export class SessionSynchronizer extends SyncMixin(SessionSyncBase) {}

export default SessionSynchronizer;
