import { select } from 'redux-saga/effects'
import { getCreatedNotes, getUpdatedNotes, getDeletedNotes } from './selectors';
import NoteRecord from '../store/noteRecord';
import { doDeleteNote, loadNote, resetActionNote } from '../actions/notesActions';
import Immutable from 'immutable';
import SyncMixin from './BaseSyncronizer';
import WebAnnotationSerializer from '../api/WebAnnotationSerializer';

class NoteSyncBase {

  // local diffs (immutable)
  * getLocalDiffs() {
    return Immutable.Map({
      created: yield select(getCreatedNotes),
      updated: yield select(getUpdatedNotes),
      deleted: yield select(getDeletedNotes)
    })
  }

  // remote urls
  getRemoteLoadUrl() {
    return "/api/notes/notes/";
  }

  getRemoteDeleteUrl(localObjInst) {
    return `/api/notes/sessions/${localObjInst.get('session')}/notes/${localObjInst.get('_id')}/`;
  }

  getRemoteCreateUrl(localObjInst) {
    return `/api/notes/sessions/${localObjInst.get('session')}/notes/`;
  }

  getRemoteUpdateUrl(localObjInst) {
    return `/api/notes/sessions/${localObjInst.get('session')}/notes/${localObjInst.get('_id')}/`;
  }

  // build remote json message
  getRemoteData(localObjInst) {

    return {
      ext_id: localObjInst.get('_id'),
      session: localObjInst.get('session'),
      raw: JSON.stringify(localObjInst.get('raw')),
      plain: localObjInst.get('plain'),
      html: localObjInst.get('html'),
      tc_start: localObjInst.get('startedAt'),
      tc_end: localObjInst.get('finishedAt'),
      categorization: JSON.stringify(WebAnnotationSerializer.serialize(localObjInst)),
      margin_note: localObjInst.get('marginComment'),
    }

  }

  getLocalRecord(remoteObj) {
    return new NoteRecord({
      _id: remoteObj.ext_id,
      session: remoteObj.session,
      raw: JSON.parse(remoteObj.raw),
      plain: remoteObj.plain,
      html: remoteObj.html,
      startedAt: remoteObj.tc_start,
      finishedAt: remoteObj.tc_end,
      categories: remoteObj.categorization,
      marginComment: remoteObj.margin_note,
    });
  }

  // actions
  doDeleteLocalObj(localObjId) {
    return doDeleteNote(localObjId);
  }

  resetLocalObj(localObjInst) {
    return resetActionNote(localObjInst);
  }

  loadObj(objRecord) {
    return loadNote(objRecord);
  }

}

export class NoteSynchronizer extends SyncMixin(NoteSyncBase) {}

export default NoteSynchronizer;
