import {
  getSessions,
  getNotes,
  getSession,
  getSessionNotes,
  getUpdatedSessions,
  getCreatedSessions,
  getDeletedSessions,
  getUpdatedNotes,
  getCreatedNotes,
  getDeletedNotes,
  getActiveSessions
} from '../coreSelectors';

import { ActionEnum } from '../../constants';

describe('Core selector test', () => {

  test('getSessions', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      sessions: [ { title: 'session1', action: ActionEnum.NONE }, { title: 'session2', action: ActionEnum.NONE }, { title: 'session3', action: ActionEnum.NONE } ]
    };

    expect(getSessions(inputState)).toEqual([ { title: 'session1', action: ActionEnum.NONE }, { title: 'session2', action: ActionEnum.NONE }, { title: 'session3', action: ActionEnum.NONE } ]);
  });

  test('getNotes', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      notes: [ { title: 'note1', action: ActionEnum.NONE }, { title: 'note2', action: ActionEnum.NONE }, { title: 'note3', action: ActionEnum.NONE } ]
    };

    expect(getNotes(inputState)).toEqual([ { title: 'note1', action: ActionEnum.NONE }, { title: 'note2', action: ActionEnum.NONE }, { title: 'note3', action: ActionEnum.NONE } ]);
  });

  test('getSessionOk', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      sessions: [ { _id: 'id1', title: 'session1', action: ActionEnum.NONE }, { _id: 'id2', title: 'session2', action: ActionEnum.NONE }, { _id: 'id3', title: 'session3', action: ActionEnum.NONE } ]
    };

    expect(getSession('id2',inputState)).toEqual({ _id: 'id2', title: 'session2', action: ActionEnum.NONE });
  });

  test('getSessionKo', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      sessions: [ { _id: 'id1', title: 'session1', action: ActionEnum.NONE }, { _id: 'id2', title: 'session2', action: ActionEnum.NONE }, { _id: 'id3', title: 'session3', action: ActionEnum.NONE } ]
    };

    expect(getSession('id0',inputState)).toEqual(undefined);
  });


  test('getSessionNotes', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      notes: [
        {
          _id: 'idNote1',
          session: "sessionId1",
          title: 'note1',
          action: ActionEnum.NONE,
          startedAt: 100,
        }, {
          _id: 'idNote2',
          session: "sessionId2",
          title: 'note2',
          action: ActionEnum.NONE,
          startedAt : 200,
        }, {
          _id: 'idNote3',
          session: "sessionId1",
          title: 'note3',
          action: ActionEnum.NONE,
          startedAt: 300,
        }, {
          _id: 'idNote4',
          session: "sessionId1",
          title: 'note4',
          action: ActionEnum.NONE,
          startedAt: 150,
        }, {
          _id: 'idNote5',
          session: "sessionId0",
          title: 'note5',
          action: ActionEnum.DELETED,
          startedAt: 400,
        }, {
          _id: 'idNote6',
          session: "sessionId1",
          title: 'note6',
          action: ActionEnum.UPDATED,
          startedAt: 250,
        }, {
          _id: 'idNote7',
          session: "sessionId1",
          title: 'note7',
          action: ActionEnum.DELETED,
          startedAt: 500,
        }, {
          _id: 'idNote8',
          session: "sessionId1",
          title: 'note8',
          action: ActionEnum.NONE,
          startedAt: 50,
        }
      ]
    };

    expect(getSessionNotes('sessionId1',inputState)).toEqual([
      {
        _id: 'idNote8',
        session: "sessionId1",
        title: 'note8',
        action: ActionEnum.NONE,
        startedAt: 50,
      }, {
        _id: 'idNote1',
        session: "sessionId1",
        title: 'note1',
        action: ActionEnum.NONE,
        startedAt: 100,
      }, {
        _id: 'idNote4',
        session: "sessionId1",
        title: 'note4',
        action: ActionEnum.NONE,
        startedAt: 150,
      }, {
        _id: 'idNote6',
        session: "sessionId1",
        title: 'note6',
        action: ActionEnum.UPDATED,
        startedAt: 250,
      }, {
        _id: 'idNote3',
        session: "sessionId1",
        title: 'note3',
        action: ActionEnum.NONE,
        startedAt: 300,
      },
    ]);
  });

  test('getUpdatedSessions', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      sessions: [
        { _id: 'sessionId1', title: 'session1', action: ActionEnum.UPDATED }, { _id: 'sessionId2', title: 'session2', action: ActionEnum.NONE },
        { _id: 'sessionId3', title: 'session3', action: ActionEnum.CREATED }, { _id: 'sessionId4', title: 'session4', action: ActionEnum.UPDATED },
        { _id: 'sessionId5', title: 'session5', action: ActionEnum.DELETED }, { _id: 'sessionId6', title: 'session6', action: ActionEnum.UPDATED }
      ]
    };

    expect(getUpdatedSessions(inputState)).toEqual( {
      'sessionId1': { _id: 'sessionId1', title: 'session1', action: ActionEnum.UPDATED },
      'sessionId4': { _id: 'sessionId4', title: 'session4', action: ActionEnum.UPDATED },
      'sessionId6': { _id: 'sessionId6', title: 'session6', action: ActionEnum.UPDATED }
    });
  });

  test('getCreatedSessions', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      sessions: [
        { _id: 'sessionId1', title: 'session1', action: ActionEnum.UPDATED }, { _id: 'sessionId2', title: 'session2', action: ActionEnum.NONE },
        { _id: 'sessionId3', title: 'session3', action: ActionEnum.CREATED }, { _id: 'sessionId4', title: 'session4', action: ActionEnum.UPDATED },
        { _id: 'sessionId5', title: 'session5', action: ActionEnum.DELETED }, { _id: 'sessionId6', title: 'session6', action: ActionEnum.CREATED }
      ]
    };

    expect(getCreatedSessions(inputState)).toEqual( {
      'sessionId3': { _id: 'sessionId3', title: 'session3', action: ActionEnum.CREATED },
      'sessionId6': { _id: 'sessionId6', title: 'session6', action: ActionEnum.CREATED }
    });
  });

  test('getDeletedSessions', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      sessions: [
        { _id: 'sessionId1', title: 'session1', action: ActionEnum.UPDATED }, { _id: 'sessionId2', title: 'session2', action: ActionEnum.NONE },
        { _id: 'sessionId3', title: 'session3', action: ActionEnum.CREATED }, { _id: 'sessionId4', title: 'session4', action: ActionEnum.UPDATED },
        { _id: 'sessionId5', title: 'session5', action: ActionEnum.DELETED }, { _id: 'sessionId6', title: 'session6', action: ActionEnum.CREATED }
      ]
    };

    expect(getDeletedSessions(inputState)).toEqual( {
      'sessionId5': { _id: 'sessionId5', title: 'session5', action: ActionEnum.DELETED }
    });
  });

  test('getActiveSessions', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      sessions: [
        { _id: 'sessionId1', title: 'session1', action: ActionEnum.UPDATED }, { _id: 'sessionId2', title: 'session2', action: ActionEnum.NONE },
        { _id: 'sessionId3', title: 'session3', action: ActionEnum.CREATED }, { _id: 'sessionId4', title: 'session4', action: ActionEnum.UPDATED },
        { _id: 'sessionId5', title: 'session5', action: ActionEnum.DELETED }, { _id: 'sessionId6', title: 'session6', action: ActionEnum.CREATED }
      ]
    };

    expect(getActiveSessions(inputState)).toEqual([
      { _id: 'sessionId1', title: 'session1', action: ActionEnum.UPDATED }, { _id: 'sessionId2', title: 'session2', action: ActionEnum.NONE },
      { _id: 'sessionId3', title: 'session3', action: ActionEnum.CREATED }, { _id: 'sessionId4', title: 'session4', action: ActionEnum.UPDATED },
      { _id: 'sessionId6', title: 'session6', action: ActionEnum.CREATED }
    ]);
  });

  test('getUpdatedNotes', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      sessions: [
        { _id: 'sessionId1', title: 'session1', action: ActionEnum.UPDATED }, { _id: 'sessionId2', title: 'session2', action: ActionEnum.NONE },
        { _id: 'sessionId3', title: 'session3', action: ActionEnum.CREATED }, { _id: 'sessionId4', title: 'session4', action: ActionEnum.UPDATED },
        { _id: 'sessionId5', title: 'session5', action: ActionEnum.DELETED }, { _id: 'sessionId6', title: 'session6', action: ActionEnum.CREATED }
      ],
      notes: [
        {
          _id: 'idNote1',
          session: "sessionId1",
          title: 'note1',
          action: ActionEnum.UPDATED,
          startedAt: 100,
        }, {
          _id: 'idNote2',
          session: "sessionId2",
          title: 'note2',
          action: ActionEnum.UPDATED,
          startedAt : 200,
        }, {
          _id: 'idNote3',
          session: "sessionId1",
          title: 'note3',
          action: ActionEnum.NONE,
          startedAt: 300,
        }, {
          _id: 'idNote4',
          session: "sessionId5",
          title: 'note4',
          action: ActionEnum.UPDATED,
          startedAt: 150,
        }, {
          _id: 'idNote5',
          session: "sessionId6",
          title: 'note5',
          action: ActionEnum.DELETED,
          startedAt: 400,
        }, {
          _id: 'idNote6',
          session: "sessionId3",
          title: 'note6',
          action: ActionEnum.UPDATED,
          startedAt: 250,
        }, {
          _id: 'idNote7',
          session: "sessionId6",
          title: 'note7',
          action: ActionEnum.DELETED,
          startedAt: 500,
        }, {
          _id: 'idNote8',
          session: "sessionId5",
          title: 'note8',
          action: ActionEnum.UPDATED,
          startedAt: 50,
        }
      ]
    };

    expect(getUpdatedNotes(inputState)).toEqual({
      'idNote1': {
        _id: 'idNote1',
        session: "sessionId1",
        title: 'note1',
        action: ActionEnum.UPDATED,
        startedAt: 100,
      },
      'idNote2': {
        _id: 'idNote2',
        session: "sessionId2",
        title: 'note2',
        action: ActionEnum.UPDATED,
        startedAt : 200,
      },
      'idNote6': {
        _id: 'idNote6',
        session: "sessionId3",
        title: 'note6',
        action: ActionEnum.UPDATED,
        startedAt: 250,
      }
    });
  });

  test('getCreatedNotes', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      sessions: [
        { _id: 'sessionId1', title: 'session1', action: ActionEnum.UPDATED }, { _id: 'sessionId2', title: 'session2', action: ActionEnum.NONE },
        { _id: 'sessionId3', title: 'session3', action: ActionEnum.CREATED }, { _id: 'sessionId4', title: 'session4', action: ActionEnum.UPDATED },
        { _id: 'sessionId5', title: 'session5', action: ActionEnum.DELETED }, { _id: 'sessionId6', title: 'session6', action: ActionEnum.CREATED }
      ],
      notes: [
        {
          _id: 'idNote1',
          session: "sessionId1",
          title: 'note1',
          action: ActionEnum.CREATED,
          startedAt: 100,
        }, {
          _id: 'idNote2',
          session: "sessionId2",
          title: 'note2',
          action: ActionEnum.CREATED,
          startedAt : 200,
        }, {
          _id: 'idNote3',
          session: "sessionId1",
          title: 'note3',
          action: ActionEnum.NONE,
          startedAt: 300,
        }, {
          _id: 'idNote4',
          session: "sessionId5",
          title: 'note4',
          action: ActionEnum.CREATED,
          startedAt: 150,
        }, {
          _id: 'idNote5',
          session: "sessionId6",
          title: 'note5',
          action: ActionEnum.DELETED,
          startedAt: 400,
        }, {
          _id: 'idNote6',
          session: "sessionId3",
          title: 'note6',
          action: ActionEnum.CREATED,
          startedAt: 250,
        }, {
          _id: 'idNote7',
          session: "sessionId6",
          title: 'note7',
          action: ActionEnum.DELETED,
          startedAt: 500,
        }, {
          _id: 'idNote8',
          session: "sessionId5",
          title: 'note8',
          action: ActionEnum.UPDATED,
          startedAt: 50,
        }
      ]
    };

    expect(getCreatedNotes(inputState)).toEqual({
      'idNote1': {
        _id: 'idNote1',
        session: "sessionId1",
        title: 'note1',
        action: ActionEnum.CREATED,
        startedAt: 100,
      },
      'idNote2': {
        _id: 'idNote2',
        session: "sessionId2",
        title: 'note2',
        action: ActionEnum.CREATED,
        startedAt : 200,
      },
      'idNote6': {
        _id: 'idNote6',
        session: "sessionId3",
        title: 'note6',
        action: ActionEnum.CREATED,
        startedAt: 250,
      }
    });
  });

  test('getDeletedNotes', ()  => {
    const inputState = {
      status: {
        isSynchronizing: false,
        online: false
      },
      authStatus: {
        token: '',
        isAuthenticated: false,
      },
      sessions: [
        { _id: 'sessionId1', title: 'session1', action: ActionEnum.UPDATED }, { _id: 'sessionId2', title: 'session2', action: ActionEnum.NONE },
        { _id: 'sessionId3', title: 'session3', action: ActionEnum.CREATED }, { _id: 'sessionId4', title: 'session4', action: ActionEnum.UPDATED },
        { _id: 'sessionId5', title: 'session5', action: ActionEnum.DELETED }, { _id: 'sessionId6', title: 'session6', action: ActionEnum.CREATED }
      ],
      notes: [
        {
          _id: 'idNote1',
          session: "sessionId1",
          title: 'note1',
          action: ActionEnum.CREATED,
          startedAt: 100,
        }, {
          _id: 'idNote2',
          session: "sessionId2",
          title: 'note2',
          action: ActionEnum.CREATED,
          startedAt : 200,
        }, {
          _id: 'idNote3',
          session: "sessionId1",
          title: 'note3',
          action: ActionEnum.NONE,
          startedAt: 300,
        }, {
          _id: 'idNote4',
          session: "sessionId5",
          title: 'note4',
          action: ActionEnum.DELETED,
          startedAt: 150,
        }, {
          _id: 'idNote5',
          session: "sessionId6",
          title: 'note5',
          action: ActionEnum.DELETED,
          startedAt: 400,
        }, {
          _id: 'idNote6',
          session: "sessionId3",
          title: 'note6',
          action: ActionEnum.CREATED,
          startedAt: 250,
        }, {
          _id: 'idNote7',
          session: "sessionId6",
          title: 'note7',
          action: ActionEnum.DELETED,
          startedAt: 500,
        }, {
          _id: 'idNote8',
          session: "sessionId5",
          title: 'note8',
          action: ActionEnum.UPDATED,
          startedAt: 50,
        }
      ]
    };

    expect(getDeletedNotes(inputState)).toEqual({
      'idNote5': {
        _id: 'idNote5',
        session: "sessionId6",
        title: 'note5',
        action: ActionEnum.DELETED,
        startedAt: 400,
      },
      'idNote7': {
        _id: 'idNote7',
        session: "sessionId6",
        title: 'note7',
        action: ActionEnum.DELETED,
        startedAt : 500,
      }
    });
  });

})
