import logging

from django.contrib.auth import get_user_model
from django.contrib.auth.models import Group
from rest_framework import serializers

logger = logging.getLogger(__name__)

User = get_user_model()

class GroupSerializer(serializers.ModelSerializer):
    owner = serializers.CharField(source='profile.owner.username', read_only=True)
    description = serializers.CharField(source='profile.description')

    class Meta:
        model = Group
        fields = ['name', 'owner', 'description']


class DetailGroupSerializer(GroupSerializer):
    users = serializers.SlugRelatedField(
        many=True,
        read_only=True,
        slug_field='username',
        source='user_set'
    )

    class Meta:
        model = Group
        fields = ['name', 'owner', 'description', 'users']

class WriteGroupSerializer(serializers.ModelSerializer):
    '''
    Serializers for writing groups.
    '''

    description = serializers.CharField(source='profile.description')
    users = serializers.SlugRelatedField(
        many=True,
        slug_field='username',
        source='user_set',
        queryset=User.objects.all(),
        default=[]
    )

    class Meta:
        model = Group
        fields = ['name', 'description', 'users']


    def create(self, validated_data):
        profile_data = validated_data.pop('profile', None)

        group = super().create(validated_data)

        if profile_data is not None:
            group.profile.description = profile_data.get('description')

        user = None
        request = self.context.get("request")
        if request and hasattr(request, "user"):
            user = request.user
            group.user_set.add(user)
            group.save()
        group.profile.owner = user
        group.profile.save()

        return group

    def update(self, instance, validated_data):
        profile_data = validated_data.pop('profile', None)
        group = super().update(instance, validated_data)

        if profile_data is not None:
            group.profile.description = profile_data.get('description')
            group.profile.save()

        # check that owner is still in user list
        if group.profile.owner and group.profile.owner not in group.user_set.all():
            group.user_set.add(group.profile.owner)

        return group
