import rootReducer from '../reducers';
import rootAuthSaga from '../sagas/authSaga';
import rootGroupSaga from '../sagas/groupSaga';
import rootSyncSaga from '../sagas/syncSaga';
import networkSaga from '../sagas/networkSaga';
import { compose, createStore, applyMiddleware } from 'redux';
import { routerMiddleware } from 'react-router-redux';
import createSagaMiddleware from 'redux-saga'
import Immutable from 'immutable';
import {persistStore, autoRehydrate} from 'redux-persist-immutable'
import localForage from 'localforage';
import immutableTransform from 'redux-persist-transform-immutable';
import NoteRecord from './noteRecord';
import SessionRecord from './sessionRecord';
import UserRecord from './userRecord';
import GroupRecord from './groupRecord';
import APIClient from '../api/APIClient';
import config from '../config';
import asyncRequest from '../constants/asyncRequest';

const composeEnhancers = (process.env.NODE_ENV !== 'production' && window.__REDUX_DEVTOOLS_EXTENSION_COMPOSE__) ?
    window.__REDUX_DEVTOOLS_EXTENSION_COMPOSE__({
        shouldHotReload: false,
    }) : compose;


const defaultState = {
  sessions: Immutable.List([]),
  notes: Immutable.List([]),
  groups: Immutable.List([]),
  status: Immutable.Map({
    isSynchronizing: false,
    online: false
  }),
  authStatus: Immutable.Map({
    token: '',
    isAuthenticated: false,
    clientId: null,
    lastSync: 0,
    currentUser: null,
  }),
  autoSubmit: false,
  login: asyncRequest,
  register: asyncRequest,
  createGroup: asyncRequest
};

const immutableTransformConfig = {
  records: [NoteRecord, SessionRecord, UserRecord, GroupRecord],
  whitelist: ['sessions', 'notes', 'authStatus', 'groups']
}

const persistOptions = {
  storage: localForage,
  records: [NoteRecord, SessionRecord, UserRecord, GroupRecord],
  transforms: [immutableTransform(immutableTransformConfig)],
  whitelist: ['sessions', 'notes', 'autoSubmit', 'authStatus', 'groups']
}

const apiClient = new APIClient(config.apiRootUrl);

const storeInitialState = Immutable.Map({ ...defaultState });


export default (history, initialState = storeInitialState) => {

  const router = routerMiddleware(history);
  const saga = createSagaMiddleware();

  const store = createStore(rootReducer, initialState, composeEnhancers(
    applyMiddleware(router, saga),
    autoRehydrate()
  ));

  apiClient.setStore(store);

  const context = {
    client: apiClient,
    history
  }

  persistStore(store, persistOptions);

  saga.run(rootAuthSaga, context);
  saga.run(rootGroupSaga, context);
  saga.run(rootSyncSaga, context);
  saga.run(networkSaga, context);

  return store;
};
