import Immutable from 'immutable';
import * as types from '../constants/actionTypes';
import UserRecord from '../store/userRecord';
import GroupRecord from '../store/groupRecord';
import asyncRequest from '../constants/asyncRequest';
import uuidV4 from 'uuid/v4';
import _ from 'lodash';

export const isAuthenticated = (state = false, action) => {
  switch (action.type) {
    case types.AUTH_DEAUTHENTICATE:
    case types.AUTH_LOGOUT:
      return false;
    case types.AUTH_LOGIN_SUCCESS:
      return true;
    default:
      return state;
  }
}

export const currentUser = (state = null, action) => {
  switch (action.type) {
    case types.AUTH_DEAUTHENTICATE:
    case types.AUTH_LOGOUT:
      return null;
    case types.AUTH_LOGIN_SUCCESS:
      return new UserRecord(action.user);
    case types.USER_UPDATE_SETTINGS:
      return state.merge({
        first_name: action.firstname,
        last_name: action.lastname
      });
    default:
      return state;
  }
}

export const token = (state = null, action) => {
  switch (action.type) {
    case types.AUTH_DEAUTHENTICATE:
    case types.AUTH_LOGOUT:
      return null;
    case types.AUTH_STORE_TOKEN:
      return action.token;
    default:
      return state;
  }
}

export const clientId = (state = null, action) => {
  switch (action.type) {
    case types.AUTH_DEAUTHENTICATE:
    case types.AUTH_LOGOUT:
      return null;
    case types.AUTH_LOGIN_SUCCESS:
      return uuidV4();
    default:
      return state;
  }

}

export const login = (state = asyncRequest, action) => {
  switch (action.type) {
    case types.AUTH_LOGIN_REQUEST:
      return Immutable.Map({
        loading: true,
        success: false,
        error: false,
      })
    case types.AUTH_LOGIN_SUCCESS:
    case types.AUTH_LOGIN_ERROR:
      return Immutable.Map({
        loading: false,
        success: action.type === types.AUTH_LOGIN_SUCCESS,
        error: action.type === types.AUTH_LOGIN_ERROR,
        errorMessages: action.type === types.AUTH_LOGIN_ERROR ? Immutable.Map(action.error) : Immutable.Map({})
      })
    default:
      return state
  }
}

export const register = (state = asyncRequest, action) => {
  switch (action.type) {
    case types.AUTH_REGISTER_REQUEST:
      return Immutable.Map({
        loading: true,
        success: false,
        error: false,
      })
    case types.AUTH_LOGIN_SUCCESS:
    case types.AUTH_REGISTER_ERROR:
      return Immutable.Map({
        loading: false,
        success: action.type === types.AUTH_LOGIN_SUCCESS,
        error: action.type === types.AUTH_REGISTER_ERROR,
        errorMessages: action.type === types.AUTH_REGISTER_ERROR ? Immutable.Map(action.error) : Immutable.Map({})
      })
    default:
      return state
  }
}

export const groups = (state = Immutable.List([]), action) => {
  switch (action.type) {
    case types.GROUP_LOAD_SUCCESS:
      return Immutable.List(
        action.groups.map((group) => GroupRecord({
          ...(_.omit(group, 'is_personal')),
          isPersonal: group['is_personal']
        }))
      );
    case types.GROUP_CREATE_SUCCESS:
      return state.push(GroupRecord({
          ...(_.omit(action.group, 'is_personal')),
          isPersonal: action.group['is_personal']
        }));
    case types.AUTH_LOGOUT: {
      return Immutable.List(); // empty note list on logout
    }
    default:
      return state
  }
}

export const createGroup = (state = asyncRequest, action) => {
  switch (action.type) {
    case types.GROUP_CREATE_ASYNC:
      return Immutable.Map({
        loading: true,
        success: false,
        error: false,
      })
    case types.GROUP_CREATE_SUCCESS:
    case types.GROUP_CREATE_ERROR:
      return Immutable.Map({
        loading: false,
        success: action.type === types.GROUP_CREATE_SUCCESS,
        error: action.type === types.GROUP_CREATE_ERROR,
        errorMessages: action.type === types.GROUP_CREATE_ERROR ? Immutable.Map(action.error) : Immutable.Map({})
      })
    default:
      return state
  }
}
