"""
Serializers for model core classes
"""
import logging

from django.contrib.auth.models import Group
from rest_framework import serializers

from notes.models import Note, Session

logger = logging.getLogger(__name__)


class DetailNoteSerializer(serializers.ModelSerializer):
    class Meta:
        model = Note
        fields = (
            'ext_id', 'version', 'created', 'updated',
            'plain', 'html', 'raw',
            'categorization', 'margin_note', 'tc_start', 'tc_end'
        )
        read_only_fields = ('ext_id', 'version', 'created', 'updated')

class UpdateNoteSerializer(serializers.ModelSerializer):
    class Meta:
        model = Note
        fields = (
            'ext_id', 'version', 'created', 'updated',
            'plain', 'html', 'raw',
            'categorization', 'margin_note', 'tc_start', 'tc_end'
        )
        read_only_fields = ('ext_id', 'version', 'created', 'updated', 'tc_start', 'tc_end')

class CreateNoteSerializer(serializers.ModelSerializer):
    class Meta:
        model = Note
        fields = (
            'ext_id', 'version', 'created', 'updated',
            'plain', 'html', 'raw',
            'categorization', 'margin_note', 'tc_start', 'tc_end'
        )
        read_only_fields = ('version', 'created', 'updated')

    def to_internal_value(self, data):
        super_data = super().to_internal_value(data)
        super_data['session'] = Session.objects.get(
            ext_id=self.context['view'].kwargs['session_ext_id']
        )
        return super_data

class ListNoteSerializer(serializers.ModelSerializer):
    class Meta:
        model = Note
        fields = (
            'ext_id', 'tc_start', 'tc_end'
        )
        read_only_fields = ('ext_id', )

class RootListNoteSerializer(serializers.ModelSerializer):
    session = serializers.SlugRelatedField(read_only=True, slug_field='ext_id')

    class Meta:
        model = Note
        fields = (
            'ext_id', 'tc_start', 'tc_end', 'session'
        )
        read_only_fields = ('ext_id', )

class RootDetailNoteSerializer(serializers.ModelSerializer):
    session = serializers.SlugRelatedField(read_only=True, slug_field='ext_id')

    class Meta:
        model = Note
        fields = (
            'ext_id', 'version', 'created', 'updated',
            'plain', 'html', 'raw',
            'categorization', 'margin_note', 'tc_start', 'tc_end',
            'session'
        )
        read_only_fields = (
            'ext_id', 'version', 'created', 'updated',
            'plain', 'html', 'raw',
            'categorization', 'margin_note', 'tc_start', 'tc_end',
            'session'
        )


class ListSessionSerializer(serializers.ModelSerializer):

    owner = serializers.SlugRelatedField(
        read_only=True, slug_field='username', default=serializers.CurrentUserDefault())
    group = serializers.SlugRelatedField(read_only=True, slug_field='name')

    class Meta:
        model = Session
        fields = (
            'ext_id', 'version', 'date', 'created', 'updated',
            'owner', 'title', 'description', 'protocol', 'group'
        )
        read_only_fields = ('ext_id', 'version', 'created', 'updated', 'owner', 'group')


class DetailSessionSerializer(serializers.ModelSerializer):

    owner = serializers.SlugRelatedField(read_only=True, slug_field='username')
    notes = DetailNoteSerializer(many=True, read_only=True)
    group = serializers.SlugRelatedField(slug_field='name', read_only=True)

    class Meta:
        model = Session
        fields = (
            'ext_id', 'version', 'date', 'created', 'updated',
            'owner', 'title', 'description', 'protocol', 'group',
            'notes'
        )
        read_only_fields = ('ext_id', 'version', 'created', 'updated', 'owner', 'group')

class CreateSessionSerializer(serializers.ModelSerializer):

    owner = serializers.SlugRelatedField(
        read_only=True, slug_field='username', default=serializers.CurrentUserDefault())
    group = serializers.SlugRelatedField(slug_field='name', queryset=Group.objects.all(), required=False)

    class Meta:
        model = Session
        fields = (
            'ext_id', 'version', 'date', 'created', 'updated',
            'owner', 'title', 'description', 'protocol', 'group'
        )
        read_only_fields = ('version', 'created', 'updated', 'owner')

    def validate(self, data):
        data = super().validate(data)

        group = data.get('group')
        owner = data.get('owner')

        if group is None:
            if owner and owner.profile and owner.profile.default_group:
                group = owner.profile.default_group
            if group is None and owner:
                group = Group.objects.filter(profile__owner_personal=owner).first()

        if group is None:
            raise serializers.ValidationError("group field is required or default group or personal group could not be found for owner")
        elif not owner in group.user_set.all():
            raise serializers.ValidationError("Owner must be in group")

        data['group'] = group
        return data

