// cf http://redux.js.org/docs/recipes/WritingTests.html

import uuidV1 from 'uuid/v1';
import * as R from 'ramda';

import reducer from '../notesReducer';
import * as types from '../../constants/ActionTypes';
import NoteRecord from '../../store/noteRecord';
import { ActionEnum } from '../../constants';

describe('note reducer', () => {
  it('should return the initial state', () => {
    expect(
      reducer(undefined, {})
    ).toEqual([])
  });

  it('should handle ADD_NOTE', () => {

    const newId = uuidV1();
    expect(
      reducer([], {
        type: types.ADD_NOTE,
        note: {
          _id: newId,
          plain: 'Run the tests'
        }
      })
    ).toEqual(
      [
        NoteRecord({
          _id: newId,
          plain: 'Run the tests',
        })
      ]
    );

    const initialStore = [
        NoteRecord({
          _id: uuidV1(),
          plain: 'original note',
        })
    ];
    expect(
      reducer(initialStore, {
        type: types.ADD_NOTE,
        note: {
          _id: newId,
          plain: 'Run the tests'
        }
      })
    ).toEqual(
      R.append(NoteRecord({
        _id: newId,
        plain: 'Run the tests',
      }),initialStore)
    );

  });

  it("Handle DELETE_NOTE", () => {

    const initialStore = [
      NoteRecord({
        _id: "id1",
        plain: 'original note',
      }),
      NoteRecord({
        _id: "id2",
        plain: 'original note 2',
      }),
      NoteRecord({
        _id: "id3",
        plain: 'original note 3',
      }),
    ];

    expect(
      reducer(initialStore, {
        type: types.DELETE_NOTE,
        note: {
          _id: "id1",
        }
      })
    ).toEqual(
      [
        NoteRecord({
          _id: "id1",
          plain: 'original note',
          action: ActionEnum.DELETED
        }),
        NoteRecord({
          _id: "id2",
          plain: 'original note 2',
        }),
        NoteRecord({
          _id: "id3",
          plain: 'original note 3',
        }),
      ]
    );

    expect(
      reducer(initialStore, {
        type: types.DELETE_NOTE,
        note: {
          _id: "id0",
        }
      })
    ).toEqual(initialStore);

    expect(
      reducer([], {
        type: types.DELETE_NOTE,
        note: {
          _id: "id0",
        }
      })
    ).toEqual([]);

  });


  it("Handle DO_DELETE_NOTE", () => {

    const initialStore = [
      NoteRecord({
        _id: "id1",
        plain: 'original note',
      }),
      NoteRecord({
        _id: "id2",
        plain: 'original note 2',
      }),
      NoteRecord({
        _id: "id3",
        plain: 'original note 3',
      }),
    ];

    expect(
      reducer(initialStore, {
        type: types.DO_DELETE_NOTE,
        noteId: "id1"
      })
    ).toEqual(
      [
        NoteRecord({
          _id: "id2",
          plain: 'original note 2',
        }),
        NoteRecord({
          _id: "id3",
          plain: 'original note 3',
        }),
      ]
    );

    expect(
      reducer(initialStore, {
        type: types.DO_DELETE_NOTE,
        noteId: "id0"
      })
    ).toEqual(initialStore);

    expect(
      reducer([], {
        type: types.DO_DELETE_NOTE,
        noteId: "id0"
      })
    ).toEqual([]);

  });


  it("Handle UPDATE_NOTE", () => {
    const initialStore = [
      NoteRecord({
        _id: "id1",
        plain: 'original note 1',
      }),
      NoteRecord({
        _id: "id2",
        plain: 'original note 2',
        action: ActionEnum.CREATED
      }),
      NoteRecord({
        _id: "id3",
        plain: 'original note 3',
        action: ActionEnum.DELETED
      }),
    ];

    expect(
      reducer(initialStore, {
        type: types.UPDATE_NOTE,
        note: NoteRecord({
          _id: "id1",
          plain: 'original note 1',
          action: ActionEnum.None
        }),
        data: {
          plain: 'new note 1'
        }
      })
    ).toEqual(
      [
        NoteRecord({
          _id: "id1",
          plain: 'new note 1',
          action: ActionEnum.UPDATED
        }),
        NoteRecord({
          _id: "id2",
          plain: 'original note 2',
          action: ActionEnum.CREATED
        }),
        NoteRecord({
          _id: "id3",
          plain: 'original note 3',
          action: ActionEnum.DELETED
        }),
      ]
    );

    expect(
      reducer(initialStore, {
        type: types.UPDATE_NOTE,
        note: NoteRecord({
          _id: "id2",
        }),
        data: {
          plain: 'new note 2'
        }
      })
    ).toEqual(
      [
        NoteRecord({
          _id: "id1",
          plain: 'original note 1'
        }),
        NoteRecord({
          _id: "id2",
          plain: 'new note 2',
          action: ActionEnum.CREATED
        }),
        NoteRecord({
          _id: "id3",
          plain: 'original note 3',
          action: ActionEnum.DELETED
        }),
      ]
    );

    expect(
      reducer(initialStore, {
        type: types.UPDATE_NOTE,
        note: NoteRecord({
          _id: "id3",
        }),
        data: {
          plain: 'new note 3'
        }
      })
    ).toEqual(
      [
        NoteRecord({
          _id: "id1",
          plain: 'original note 1'
        }),
        NoteRecord({
          _id: "id2",
          plain: 'original note 2',
          action: ActionEnum.CREATED
        }),
        NoteRecord({
          _id: "id3",
          plain: 'new note 3',
          action: ActionEnum.DELETED
        }),
      ]
    );

    expect(
      reducer(initialStore, {
        type: types.UPDATE_NOTE,
        note: {
          _id: "id0",
          plain: 'new note 0',
        }
      })
    ).toEqual(initialStore);

    expect(
      reducer([], {
        type: types.UPDATE_NOTE,
        note: {
          _id: "id0",
        }
      })
    ).toEqual([]);

  });



  it("Handle DELETE_SESSION", () => {
    const initialStore = [
      NoteRecord({
        _id: "id1",
        plain: 'original note 1',
        session: "session1"
      }),
      NoteRecord({
        _id: "id2",
        plain: 'original note 2',
        session: "session1"
      }),
      NoteRecord({
        _id: "id3",
        plain: 'original note 3',
        session: "session2"
      }),
    ];

    expect(
      reducer(initialStore, {
        type: types.DELETE_SESSION,
        session: {
          _id: "session1",
        }
      })
    ).toEqual(
      [
        NoteRecord({
          _id: "id1",
          plain: 'original note 1',
          session: "session1",
          action: ActionEnum.DELETED
        }),
        NoteRecord({
          _id: "id2",
          plain: 'original note 2',
          session: "session1",
          action: ActionEnum.DELETED
        }),
        NoteRecord({
          _id: "id3",
          plain: 'original note 3',
          session: "session2"
        }),
      ]
    );

    expect(
      reducer(initialStore, {
        type: types.DELETE_SESSION,
        session: {
          _id: "session0",
        }
      })
    ).toEqual(initialStore);

    expect(
      reducer([], {
        type: types.DELETE_SESSION,
        session: {
          _id: "session0",
        }
      })
    ).toEqual([]);

  });


  it("Handle DO_DELETE_SESSION", () => {
    const initialStore = [
      NoteRecord({
        _id: "id1",
        plain: 'original note 1',
        session: "session1"
      }),
      NoteRecord({
        _id: "id2",
        plain: 'original note 2',
        session: "session1"
      }),
      NoteRecord({
        _id: "id3",
        plain: 'original note 3',
        session: "session2"
      }),
    ];

    expect(
      reducer(initialStore, {
        type: types.DO_DELETE_SESSION,
        sessionId: "session1"
      })
    ).toEqual(
      [
        NoteRecord({
          _id: "id3",
          plain: 'original note 3',
          session: "session2"
        }),
      ]
    );

    expect(
      reducer(initialStore, {
        type: types.DO_DELETE_SESSION,
        sessionId: "session0"
      })
    ).toEqual(initialStore);

    expect(
      reducer([], {
        type: types.DO_DELETE_SESSION,
        sessionId: "session0"
      })
    ).toEqual([]);

  });


  ///
  /// Handle RESET_ACTION_NOTE
  ///
  it("Handle RESET_ACTION_NOTE", () => {

    const initialStore = [
      NoteRecord({
        _id: "id1",
        plain: 'original note 1',
        action: ActionEnum.UPDATED
      }),
      NoteRecord({
        _id: "id2",
        plain: 'original note 2',
        action: ActionEnum.UPDATED
      }),
      NoteRecord({
        _id: "id3",
        plain: 'original note 3',
        action: ActionEnum.UPDATED
      }),
    ];

    expect(
      reducer(initialStore, {
        type: types.RESET_ACTION_NOTE,
        note: {
          _id: "id1",
        }
      })
    ).toEqual(
      [
        NoteRecord({
          _id: "id1",
          plain: 'original note 1',
          action: ActionEnum.NONE
        }),
        NoteRecord({
          _id: "id2",
          plain: 'original note 2',
          action: ActionEnum.UPDATED
        }),
        NoteRecord({
          _id: "id3",
          plain: 'original note 3',
          action: ActionEnum.UPDATED
        }),
      ]
    );

    expect(
      reducer(initialStore, {
        type: types.RESET_ACTION_NOTE,
        note: {
          _id: "id0",
        }
      })
    ).toEqual(initialStore);

    expect(
      reducer([], {
        type: types.RESET_ACTION_NOTE,
        note: {
          _id: "id0",
        }
      })
    ).toEqual([]);

  });

  it("Handle SYNC_RESET_ALL", () => {

    const initialStore = [
      NoteRecord({
        _id: "id1",
        plain: 'original note 1',
        action: ActionEnum.UPDATED
      }),
      NoteRecord({
        _id: "id2",
        plain: 'original note 2',
        action: ActionEnum.UPDATED
      }),
      NoteRecord({
        _id: "id3",
        plain: 'original note 3',
        action: ActionEnum.UPDATED
      }),
    ];

    expect(
      reducer(initialStore, {
        type: types.SYNC_RESET_ALL,
      })
    ).toEqual(
      [
        NoteRecord({
          _id: "id1",
          plain: 'original note 1',
          action: ActionEnum.NONE
        }),
        NoteRecord({
          _id: "id2",
          plain: 'original note 2',
          action: ActionEnum.NONE
        }),
        NoteRecord({
          _id: "id3",
          plain: 'original note 3',
          action: ActionEnum.NONE
        }),
      ]
    );

    expect(
      reducer([], {
        type: types.SYNC_RESET_ALL,
      })
    ).toEqual([]);

  });

  it("Handle LOAD_NOTE", () => {
    expect(
      reducer([], {
        type: types.LOAD_NOTE,
        note: NoteRecord({
          _id: 'note1',
          plain: 'Run the tests'
        })
      })
    ).toEqual(
      [
        NoteRecord({
          _id: 'note1',
          plain: 'Run the tests',
        })
      ]
    );

    const initialStore = [
        NoteRecord({
          _id: 'note1',
          plain: 'original note 1',
        }),
        NoteRecord({
          _id: 'note2',
          plain: 'original note 2',
        })

    ];
    expect(
      reducer(initialStore, {
        type: types.LOAD_NOTE,
        note: NoteRecord({
          _id: 'note1',
          plain: 'Run the tests'
        })
      })
    ).toEqual(
      [
        NoteRecord({
          _id: 'note1',
          plain: 'Run the tests',
        }),
        NoteRecord({
          _id: 'note2',
          plain: 'original note 2',
        })
      ]
    );

  expect(
    reducer(initialStore, {
      type: types.LOAD_NOTE,
      note: NoteRecord({
        _id: 'note3',
        plain: 'Run the tests'
      })
    })
  ).toEqual(
    [
      NoteRecord({
        _id: 'note1',
        plain: 'original note 1',
      }),
      NoteRecord({
        _id: 'note2',
        plain: 'original note 2',
      }),
      NoteRecord({
        _id: 'note3',
        plain: 'Run the tests'
      })
    ]
  );

  });

  it("Handle AUTH_LOGOUT", () => {
    const initialStore = [
      NoteRecord({
        _id: "id1",
        plain: 'original note 1',
        action: ActionEnum.UPDATED
      }),
      NoteRecord({
        _id: "id2",
        plain: 'original note 2',
        action: ActionEnum.UPDATED
      }),
      NoteRecord({
        _id: "id3",
        plain: 'original note 3',
        action: ActionEnum.UPDATED
      }),
    ];

    expect(
      reducer(initialStore, {
        type: types.AUTH_LOGOUT,
      })
    ).toEqual(
      []
    );

    expect(
      reducer([], {
        type: types.AUTH_LOGOUT,
      })
    ).toEqual([]);

  });

});

//     expect(
//       reducer(
//         [
//           {
//             text: 'Use Redux',
//             completed: false,
//             id: 0
//           }
//         ],
//         {
//           type: types.ADD_TODO,
//           text: 'Run the tests'
//         }
//       )
//     ).toEqual(
//       [
//         {
//           text: 'Run the tests',
//           completed: false,
//           id: 1
//         },
//         {
//           text: 'Use Redux',
//           completed: false,
//           id: 0
//         }
//       ]
//     )
