import React, { Component } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import { Form, FormControl, Label } from 'react-bootstrap';
import moment from 'moment';

import PropTypes from 'prop-types';
import SlateEditor from './SlateEditor';
import * as notesActions from '../actions/notesActions';
import * as userActions from '../actions/userActions';

class NoteInput extends Component {

  state = {
    buttonDisabled: false,
    startedAt: null,
    finishedAt: null,
  }

  onEditorChange = (e) => {
    this.setState({
      buttonDisabled: e.state.document.length === 0,
      startedAt: e.startedAt,
      finishedAt: e.finishedAt
    });
  }

  onAddNoteClick = () => {

    const plain = this.refs.editor.asPlain();
    const raw = this.refs.editor.asRaw();
    const html = this.refs.editor.asHtml();
    const categories = this.refs.editor.asCategories();

    this.props.notesActions.addNote(this.props.session, {
      plain: plain,
      raw: raw,
      html: html,

      startedAt: this.state.startedAt,
      finishedAt: moment().format('H:mm:ss'),
      categories: categories,
    });

    this.refs.editor.clear();
    setTimeout(() => this.refs.editor.focus(), 250);
  }

  onCheckboxChange = (e) => {
    this.props.userActions.setAutoSubmit(e.target.checked);
  }

  componentDidMount() {
    const text = this.refs.editor.asPlain();
    this.setState({ buttonDisabled: text.length === 0 });
  }

  renderTiming() {
    if (this.state.startedAt && this.state.finishedAt) {
      return (
        <span>
          <Label>{this.state.startedAt}</Label> ⇒ <Label>{this.state.finishedAt}</Label>
        </span>
      )
    } else {
      return (
        <span>No timing</span>
      )
    }
  }

  render() {
    return (
      <Form>
        <div className="editor">
          <div className="editor-left">
            <SlateEditor ref="editor"
              onChange={this.onEditorChange}
              onEnterKeyDown={this.onAddNoteClick}
              onButtonClick={this.onAddNoteClick}
              onCheckboxChange={this.onCheckboxChange}
              isChecked={this.props.autoSubmit}
              isButtonDisabled={this.state.buttonDisabled} />
          </div>
          <div className="editor-right">
            <FormControl
              name="margin"
              componentClass="textarea"
              placeholder="Enter a margin comment for your note"
              inputRef={ ref => { this.description = ref; } }
            />
          </div>
        </div>
      </Form>
    );
  }
}

function mapStateToProps(state, props) {
  return {
    autoSubmit: state.autoSubmit
  };
}

function mapDispatchToProps(dispatch) {
  return {
    notesActions: bindActionCreators(notesActions, dispatch),
    userActions: bindActionCreators(userActions, dispatch),
  }
}

NoteInput.propTypes = {
  addNote: PropTypes.func.isRequired
};

export default connect(mapStateToProps, mapDispatchToProps)(NoteInput);
