"""
Tests the core api for sessions
"""
import datetime
import logging
from uuid import uuid4

from django.contrib.auth import get_user_model
from django.urls import reverse
from django.utils import timezone
from rest_framework import status
from rest_framework.test import APITransactionTestCase

from notes.models import Session, Note

logger = logging.getLogger(__name__)


class NoteApiTests(APITransactionTestCase):
    '''
    Test Note api
    '''

    def setUp(self):
        User = get_user_model()
        user1 = User.objects.create_user(
            username='test_user1',
            email='test_user@emial.com',
            password='top_secret'
        )
        user2 = User.objects.create_user(
            username='test_user2',
            email='test_user@emial.com',
            password='top_secret'
        )
        user3 = User.objects.create_user(
            username='test_user3',
            email='test_user@emial.com',
            password='top_secret'
        )

        self.session1 = Session.objects.create(
            title="a new session 1",
            description="Description 1",
            protocol="[]",
            owner=user1
        )

        self.session2 = Session.objects.create(
            title="a new session 2",
            description="Description 2",
            protocol="[]",
            owner=user2
        )

        Session.objects.create(
            title="a new session 3",
            description="Description 3",
            protocol="[]",
            owner=user3
        )

        self.note1 = Note.objects.create(
            tc_start=timezone.now(),
            tc_end=timezone.now(),
            session=self.session1,
            plain="example note 1",
            html="<i>example note 1</i>",
            raw="<i>example note 1</i>",
            margin_note="margin note 1",
            categorization="[]"
        )

        self.note2 = Note.objects.create(
            tc_start=timezone.now(),
            tc_end=timezone.now(),
            session=self.session1,
            plain="example note 1.1",
            html="<i>example note 1,1</i>",
            raw="<i>example note 1.1</i>",
            margin_note="margin note 1.1",
            categorization="[]"
        )

        Note.objects.create(
            tc_start=timezone.now(),
            tc_end=timezone.now(),
            session=self.session2,
            plain="example note 2",
            html="<i>example note</i>",
            raw="<i>example note</i>",
            margin_note="margin note",
            categorization="[]"
        )

    def test_create_note_no_user(self):
        url = reverse('notes:notes-session-list',
                      kwargs={'session_ext_id': self.session1.ext_id})
        response = self.client.post(url, {
            'tc_start': timezone.now(),
            'tc_end': timezone.now(),
            'plain': "example note 2",
            'html': "<i>example note</i>",
            'raw': "<i>example note</i>",
            'margin_note': "margin note",
            'categorization': "[]"
        }, format='json')

        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED)

    def test_create_note(self):
        url = reverse('notes:notes-session-list',
                      kwargs={'session_ext_id': self.session1.ext_id})
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.post(url, {
            'tc_start': timezone.now(),
            'tc_end': timezone.now(),
            'plain': "example note 2",
            'html': "<i>example note</i>",
            'raw': "<i>example note</i>",
            'margin_note': "margin note",
            'categorization': "[]"
        }, format='json')

        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        json = response.json()
        self.assertIn('ext_id', json)
        note = Note.objects.get(
            ext_id=json['ext_id'], session__id=self.session1.id)
        self.assertTrue(note)

    def test_create_note_with_ext_id(self):
        url = reverse('notes:notes-session-list',
                      kwargs={'session_ext_id': self.session1.ext_id})
        self.client.login(username='test_user1', password='top_secret')
        ext_id = str(uuid4())
        response = self.client.post(url, {
            'ext_id': ext_id,
            'tc_start': timezone.now(),
            'tc_end': timezone.now(),
            'plain': "example note 2",
            'html': "<i>example note</i>",
            'raw': "<i>example note</i>",
            'margin_note': "margin note",
            'categorization': "[]"
        }, format='json')

        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        json = response.json()
        self.assertIn('ext_id', json)
        self.assertEqual(json['ext_id'], ext_id)
        note = Note.objects.get(ext_id=ext_id)
        self.assertTrue(note)

    def test_update_note(self):
        url = reverse('notes:notes-session-detail',
                      kwargs={'session_ext_id': self.session1.ext_id, 'ext_id': self.note1.ext_id})
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.put(url, {
            'plain': "example note 1 modified",
            'html': "<i>example note modified</i>",
            'raw': "<i>example note modified</i>",
            'margin_note': "margin note",
            'categorization': "[]"
        }, format='json')

        self.assertEqual(response.status_code, status.HTTP_200_OK)
        json = response.json()
        self.assertIn('plain', json)
        self.assertEqual("example note 1 modified", json['plain'])
        note = Note.objects.get(
            ext_id=json['ext_id'], session__id=self.session1.id)
        self.assertTrue(note)
        self.assertEqual("example note 1 modified", note.plain)

    # TODO: Fail if a tc_start, tc_end, session, ext_id updated, created are provided on update ?
    # def test_update_note_tc_start(self):
    #     url = reverse('notes:notes-session-detail',
    #                   kwargs={'session_ext_id': self.session1.ext_id, 'ext_id': self.note1.ext_id})
    #     self.client.login(username='test_user1', password='top_secret')
    #     response = self.client.put(url, {
    #         'tc_start': timezone.now(),
    #         'tc_end': timezone.now(),
    #         'plain': "example note 1 modified",
    #         'html': "<i>example note modified</i>",
    #         'raw': "<i>example note modified</i>",
    #         'margin_note': "margin note",
    #         'categorization': "[]"
    #     }, format='json')

    #     self.assertEqual(response.status_code, status.HTTP_200_OK)

    def test_root_note_list(self):
        url = reverse('notes:note-list')
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url, format='json')
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        json_resp = response.json()
        self.assertIn('results', json_resp)
        self.assertEqual(2, json_resp['count'])
        self.assertEqual(2, len(json_resp['results']))

        for note_json in json_resp['results']:
            self.assertEqual(str(self.session1.ext_id),
                             note_json.get('session'))
            self.assertIn('plain', note_json)
            self.assertIn('html', note_json)
            self.assertIn('raw', note_json)

    def test_root_note_list_filter(self):
        url = reverse('notes:note-list')
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url, {'ext_id__in': ",".join(
            [str(self.note1.ext_id), str(uuid4())])}, format='json')
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        json_resp = response.json()
        self.assertIn('results', json_resp)
        self.assertEqual(1, json_resp['count'])
        self.assertEqual(1, len(json_resp['results']))

        note_json = json_resp['results'][0]
        self.assertEqual(str(self.note1.ext_id), note_json.get('ext_id'))
        self.assertEqual(str(self.session1.ext_id), note_json.get('session'))
        self.assertIn('plain', note_json)
        self.assertIn('html', note_json)
        self.assertIn('raw', note_json)

    def test_root_note_list_filter_both(self):
        url = reverse('notes:note-list')
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url, {'ext_id__in': ",".join(
            [str(self.note1.ext_id), str(self.note2.ext_id)])}, format='json')
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        json_resp = response.json()
        self.assertIn('results', json_resp)
        self.assertEqual(2, json_resp['count'])
        self.assertEqual(2, len(json_resp['results']))

        for note_json in json_resp['results']:
            self.assertIn(note_json.get('ext_id'), [str(self.note1.ext_id), str(self.note2.ext_id)])
            self.assertEqual(str(self.session1.ext_id), note_json.get('session'))
            self.assertIn('plain', note_json)
            self.assertIn('html', note_json)
            self.assertIn('raw', note_json)


    def test_root_note_list_bad_filter(self):
        url = reverse('notes:note-list')
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url, {'ext_id__in': ",".join(
            [str(self.note1.ext_id), "foo"])}, format='json')
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        json_resp = response.json()
        self.assertIn('results', json_resp)
        self.assertEqual(0, json_resp['count'])
        self.assertEqual(0, len(json_resp['results']))


    def test_root_note_list_modified(self):

        nexthour = \
            datetime.datetime.utcnow().replace(tzinfo=timezone.utc) +\
            datetime.timedelta(hours=1)
        Note.objects.filter(pk=self.note2.id).update(updated=nexthour)

        url = reverse('notes:note-list')
        self.client.login(username='test_user1', password='top_secret')

        response = self.client.get(
            url,
            {'modified_since': (
                nexthour - datetime.timedelta(minutes=30)).timestamp()},
            format='json'
        )

        self.assertEqual(response.status_code, status.HTTP_200_OK)
        json_resp = response.json()
        self.assertIn('results', json_resp)
        self.assertEqual(1, json_resp['count'])

        self.assertEqual(len(json_resp['results']), 1, "must have one note")
        self.assertEqual(str(self.session1.ext_id),
                         json_resp['results'][0].get('session'))

    def test_root_note_detail(self):
        url = reverse('notes:note-detail',
                      kwargs={'ext_id': self.note2.ext_id})
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url, format='json')
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        json_resp = response.json()
        self.assertEqual(str(self.session1.ext_id), json_resp.get('session'))
        self.assertEqual('example note 1.1', json_resp.get('plain'))
