"""
Tests the core api for sessions
"""
import logging

from django.contrib.auth import get_user_model
from django.urls import reverse
from django.utils import timezone
from rest_framework import status
from rest_framework.test import APITransactionTestCase

from notes.models import Session, Note

logger = logging.getLogger(__name__)

class SessionApiTests(APITransactionTestCase):

    def setUp(self):
        User = get_user_model()
        user1 = User.objects.create_user(
            username='test_user1',
            email='test_user@emial.com',
            password='top_secret'
        )
        user2 = User.objects.create_user(
            username='test_user2',
            email='test_user@emial.com',
            password='top_secret'
        )
        user3 = User.objects.create_user(
            username='test_user3',
            email='test_user@emial.com',
            password='top_secret'
        )

        self.session1 = Session.objects.create(
            title="a new session 1",
            description="Description 1",
            protocol="[]",
            owner=user1
        )

        self.session2 = Session.objects.create(
            title="a new session 2",
            description="Description 2",
            protocol="[]",
            owner=user2
        )

        Session.objects.create(
            title="a new session 3",
            description="Description 3",
            protocol="[]",
            owner=user3
        )

        Note.objects.create(
            tc_start=timezone.now(),
            tc_end=timezone.now(),
            session=self.session1,
            plain="example note 1",
            html="<i>example note 1</i>",
            raw="<i>example note 1</i>",
            margin_note="margin note 1",
            categorization="[]"
        )

        Note.objects.create(
            tc_start=timezone.now(),
            tc_end=timezone.now(),
            session=self.session2,
            plain="example note 2",
            html="<i>example note</i>",
            raw="<i>example note</i>",
            margin_note="margin note",
            categorization="[]"
        )


    def test_list_session_no_user(self):
        url = reverse('notes_api:session-list')
        response = self.client.post(url)
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)


    def test_list_session(self):
        url = reverse('notes_api:session-list')
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        json = response.json()
        self.assertEqual(len(json), 1, "must have one session")
        for session in json:
            self.assertEqual(session['owner'], 'test_user1')


    def test_create_session_no_user(self):
        url = reverse('notes_api:session-list')
        response = self.client.post(url, {
            'title': "a new session",
            'description': "description of the session",
            'protocol': "[]"
        }, format='json')

        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)


    def test_create_session(self):
        url = reverse('notes_api:session-list')
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.post(url, {
            'title': "a new session",
            'description': "description of the session",
            'protocol': "[]"
        }, format='json')


        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        json = response.json()
        self.assertIn('ext_id', json)

    def test_detail_session(self):
        url = reverse('notes_api:session-detail', kwargs={'ext_id':str(self.session1.ext_id)})
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url, format='json')
        self.assertEqual(response.status_code, status.HTTP_200_OK)

    def test_list_notes(self):
        url = reverse('notes_api:notes-list', kwargs={'session_ext_id':str(self.session1.ext_id)})
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url, format='json')
        self.assertEqual(response.status_code, status.HTTP_200_OK)

    def test_detail_session_bad(self):
        url = reverse('notes_api:session-detail', kwargs={'ext_id':str(self.session2.ext_id)})
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url, format='json')
        self.assertEqual(response.status_code, status.HTTP_404_NOT_FOUND)

    def test_list_notes_bad(self):
        url = reverse('notes_api:notes-list', kwargs={'session_ext_id':str(self.session2.ext_id)})
        logger.debug("URL: %s", url)
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url, format='json')
        logger.debug(response.json())
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)
