import React, { Component } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import { Panel, FormGroup, ControlLabel, FormControl, Button, InputGroup, HelpBlock } from 'react-bootstrap';
import '../App.css';
import * as sessionsActions from '../actions/sessionsActions';
import * as authActions from '../actions/authActions';
import _ from 'lodash';

class SessionForm extends Component {

  state = {
    createGroup: false
  }

  toggleCreateGroup = (e) => {
    e.preventDefault();
    const { createGroup } = this.state;
    this.setState({ createGroup: !createGroup });
  }

  onClickCreateGroup = (e) => {
    e.preventDefault();
    const groupName = this.groupName.value;
    this.props.authActions.createGroup(groupName);
  }

  onChange = (e) => {
    const { name, value } = e.target;
    const changes = { [name]: value }
    this.onChangeThrottle(changes);
  }

  onChangeThrottle = _.debounce((changes) => {
    this.props.sessionsActions.updateSession(this.props.currentSession, changes);
  }, 750)

  renderCreateGroup = () => {
    const { createGroup } = this.props;
    const hasErrors = true === createGroup.get('error') && createGroup.get('errorMessages').has('name');

    let errors = [];
    if (hasErrors) {
      const errorMessages = createGroup.get('errorMessages').toArray();
      errors = errorMessages.map((message, key) => {
        return (
          <HelpBlock key={ key }>{ message }</HelpBlock>
        )
      })
    }

    if (this.state.createGroup) {
      return (
        <FormGroup validationState={ hasErrors ? 'error' : null }>
          <InputGroup>
            <FormControl
              type="text"
              placeholder="Enter a name for your group"
              inputRef={ ref => { this.groupName = ref; } } />
            <InputGroup.Button>
              <Button bsStyle="primary" onClick={ this.onClickCreateGroup }>Create</Button>
            </InputGroup.Button>
          </InputGroup>
          { errors }
          <hr />
          <Button onClick={ this.toggleCreateGroup }>Cancel</Button>
        </FormGroup>
      )
    }

    return (
      <FormGroup>
        <Button onClick={ this.toggleCreateGroup }>Create a group</Button>
      </FormGroup>
    )
  }

  render() {

    if (!this.props.currentSession) {
      return (
        <form></form>
      )
    }

    return (
      <Panel>
        <form onSubmit={ e => { e.preventDefault() } }>
          <FormGroup>
            <ControlLabel>Title</ControlLabel>
            <FormControl
              name="title"
              defaultValue={ this.props.currentSession.title }
              onChange={ this.onChange }
              type="text"
              placeholder="Enter a title for this session"
              inputRef={ ref => { this.title = ref; } }
            />
          </FormGroup>
          <FormGroup>
            <ControlLabel>Description</ControlLabel>
            <FormControl
              name="description"
              componentClass="textarea"
              defaultValue={ this.props.currentSession.description }
              onChange={ this.onChange }
              placeholder="Enter a description for this session"
              inputRef={ ref => { this.description = ref; } }
            />
          </FormGroup>
          <FormGroup>
            { this.renderCreateGroup() }
          </FormGroup>
        </form>
      </Panel>
    );
  }
}

function mapStateToProps(state, props) {
  return {
    currentSession: props.session,
    createGroup: state.createGroup
  };
}

function mapDispatchToProps(dispatch) {
  return {
    sessionsActions: bindActionCreators(sessionsActions, dispatch),
    authActions: bindActionCreators(authActions, dispatch),
  }
}

export default connect(mapStateToProps, mapDispatchToProps)(SessionForm);
