"""
Tests the core api for sessions
"""
import logging
from uuid import uuid4

from django.contrib.auth import get_user_model
from django.urls import reverse
from django.utils import timezone
from rest_framework import status
from rest_framework.test import APITransactionTestCase

from notes.models import Session, Note

logger = logging.getLogger(__name__)

class NoteApiTests(APITransactionTestCase):

    def setUp(self):
        User = get_user_model()
        user1 = User.objects.create_user(
            username='test_user1',
            email='test_user@emial.com',
            password='top_secret'
        )
        user2 = User.objects.create_user(
            username='test_user2',
            email='test_user@emial.com',
            password='top_secret'
        )
        user3 = User.objects.create_user(
            username='test_user3',
            email='test_user@emial.com',
            password='top_secret'
        )

        self.session1 = Session.objects.create(
            title="a new session 1",
            description="Description 1",
            protocol="[]",
            owner=user1
        )

        self.session2 = Session.objects.create(
            title="a new session 2",
            description="Description 2",
            protocol="[]",
            owner=user2
        )

        Session.objects.create(
            title="a new session 3",
            description="Description 3",
            protocol="[]",
            owner=user3
        )

        Note.objects.create(
            tc_start=timezone.now(),
            tc_end=timezone.now(),
            session=self.session1,
            plain="example note 1",
            html="<i>example note 1</i>",
            raw="<i>example note 1</i>",
            margin_note="margin note 1",
            categorization="[]"
        )

        Note.objects.create(
            tc_start=timezone.now(),
            tc_end=timezone.now(),
            session=self.session2,
            plain="example note 2",
            html="<i>example note</i>",
            raw="<i>example note</i>",
            margin_note="margin note",
            categorization="[]"
        )


    def test_create_note_no_user(self):
        url = reverse('notes:notes-list', kwargs={'session_ext_id': self.session1.ext_id})
        response = self.client.post(url, {
            'tc_start': timezone.now(),
            'tc_end': timezone.now(),
            'plain': "example note 2",
            'html': "<i>example note</i>",
            'raw': "<i>example note</i>",
            'margin_note': "margin note",
            'categorization': "[]"
        }, format='json')

        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED)


    def test_create_note(self):
        url = reverse('notes:notes-list', kwargs={'session_ext_id': self.session1.ext_id})
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.post(url, {
            'tc_start': timezone.now(),
            'tc_end': timezone.now(),
            'plain': "example note 2",
            'html': "<i>example note</i>",
            'raw': "<i>example note</i>",
            'margin_note': "margin note",
            'categorization': "[]"
        }, format='json')

        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        json = response.json()
        self.assertIn('ext_id', json)
        note = Note.objects.get(ext_id=json['ext_id'], session__id=self.session1.id)
        self.assertTrue(note)

    def test_create_note_with_ext_id(self):
        url = reverse('notes:notes-list', kwargs={'session_ext_id': self.session1.ext_id})
        self.client.login(username='test_user1', password='top_secret')
        ext_id = str(uuid4())
        response = self.client.post(url, {
            'ext_id': ext_id,
            'tc_start': timezone.now(),
            'tc_end': timezone.now(),
            'plain': "example note 2",
            'html': "<i>example note</i>",
            'raw': "<i>example note</i>",
            'margin_note': "margin note",
            'categorization': "[]"
        }, format='json')

        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        json = response.json()
        self.assertIn('ext_id', json)
        self.assertEqual(json['ext_id'], ext_id)
        note = Note.objects.get(ext_id=ext_id)
        self.assertTrue(note)
