import uuidV1 from 'uuid/v1';

import {
  isAuthenticated,
  currentUser,
  currentGroup,
  token,
  clientId,
  login,
  register,
  groups,
  createGroup,
} from '../authReducer';
import asyncRequest from '../../constants/asyncRequest';

import * as types from '../../constants/ActionTypes'
import UserRecord from '../../store/userRecord';
import GroupRecord from '../../store/groupRecord';


describe('auth reducer isAuthenticated', () => {

  it('should return the initial state', () => {
    expect(
      isAuthenticated(undefined, {})
    ).toEqual(false)
  });

  it("Handle AUTH_DEAUTHENTICATE true", () => {
    expect(
      isAuthenticated(true, {
        type: types.AUTH_DEAUTHENTICATE
      })
    ).toEqual(false)
  });

  it("Handle AUTH_DEAUTHENTICATE false", () => {
    expect(
      isAuthenticated(false, {
        type: types.AUTH_DEAUTHENTICATE
      })
    ).toEqual(false)
  });

  it("Handle AUTH_LOGOUT true", () => {
    expect(
      isAuthenticated(true, {
        type: types.AUTH_LOGOUT
      })
    ).toEqual(false)
  });

  it("Handle AUTH_LOGOUT false", () => {
    expect(
      isAuthenticated(false, {
        type: types.AUTH_LOGOUT
      })
    ).toEqual(false)
  });


  it("Handle AUTH_LOGIN_SUCCESS false", () => {
    expect(
      isAuthenticated(false, {
        type: types.AUTH_LOGIN_SUCCESS
      })
    ).toEqual(true)
  });

  it("Handle AUTH_LOGIN_SUCCESS true", () => {
    expect(
      isAuthenticated(true, {
        type: types.AUTH_LOGIN_SUCCESS
      })
    ).toEqual(true)
  });

});


describe('auth reducer currentUser', () => {

  it('should return the initial state', () => {
    expect(
      currentUser(undefined, {})
    ).toEqual(null)
  });

  it('Handle AUTH_DEAUTHENTICATE', () => {
    const initalState = new UserRecord({ first_name: 'John', last_name: 'Doe'});
    expect(
      currentUser(initalState, {
        type: types.AUTH_DEAUTHENTICATE
      })
    ).toEqual(null)
  });


  it('Handle AUTH_LOGOUT', () => {
    const initalState = new UserRecord({ first_name: 'John', last_name: 'Doe'});
    expect(
      currentUser(initalState, {
        type: types.AUTH_LOGOUT
      })
    ).toEqual(null)
  });


  it('Handle AUTH_LOGIN_SUCCESS', () => {
    const initalState = null;
    expect(
      currentUser(initalState, {
        type: types.AUTH_LOGIN_SUCCESS,
        user: { first_name: 'John', last_name: 'Doe'}
      })
    ).toEqual(new UserRecord({ first_name: 'John', last_name: 'Doe'}))
  });

  it('Handle USER_UPDATE_SETTINGS', () => {
    const initalState = new UserRecord({ first_name: 'John', last_name: 'Doe'});
    expect(
      currentUser(initalState, {
        type: types.USER_UPDATE_SETTINGS,
        firstname: 'Jane',
        lastname: 'Orwell'
      })
    ).toEqual(new UserRecord({ first_name: 'Jane', last_name: 'Orwell'}))
  });

});

describe('auth reducer currentGroup', () => {

  it('should return the initial state', () => {
    expect(
      currentGroup(undefined, {})
    ).toEqual(null)
  });

  it('Handle AUTH_DEAUTHENTICATE', () => {
    expect(
      currentGroup('group1', {
        type: types.AUTH_DEAUTHENTICATE
      })
    ).toEqual(null);
  });

  it('Handle AUTH_LOGOUT', () => {
    expect(
      currentGroup('group1', {
        type: types.AUTH_LOGOUT
      })
    ).toEqual(null);
  });

  it('Handle AUTH_LOGIN_SUCCESS', () => {
    expect(
      currentGroup(null, {
        type: types.AUTH_LOGIN_SUCCESS,
        user: { first_name: 'John', last_name: 'Doe', default_group: 'group1' }
      })
    ).toEqual('group1');
  });

  it('Handle GROUP_CREATE_SUCCESS', () => {
    expect(
      currentGroup('group1', {
        type: types.GROUP_CREATE_SUCCESS,
        group: { name: 'group2' }
      })
    ).toEqual('group2');
  });

  it('Handle GROUP_SET_GROUP', () => {
    expect(
      currentGroup('group1', {
        type: types.GROUP_SET_GROUP,
        group: 'group2'
      })
    ).toEqual('group2');
  });
});

describe('auth reducer token', () => {

  it('should return the initial state', () => {
    expect(
      token(undefined, {})
    ).toEqual(null)
  });

  it('Handle AUTH_DEAUTHENTICATE', () => {
    expect(
      token('token1', {
        type: types.AUTH_DEAUTHENTICATE
      })
    ).toEqual(null);
  });

  it('Handle AUTH_LOGOUT', () => {
    expect(
      token('token1', {
        type: types.AUTH_LOGOUT
      })
    ).toEqual(null);

  });

  it('Handle AUTH_STORE_TOKEN', () => {
    expect(
      token(null, {
        type: types.AUTH_STORE_TOKEN,
        token: 'token1'
      })
    ).toEqual('token1');
  });
});

describe('auth reducer clientId', () => {

  it('should return the initial state', () => {
    expect(
      clientId(undefined, {})
    ).toEqual(null)
  });

  it('Handle AUTH_DEAUTHENTICATE', () => {
    expect(
      clientId(uuidV1(), {
        type: types.AUTH_DEAUTHENTICATE
      })
    ).toEqual(null);
  });

  it('Handle AUTH_LOGOUT', () => {
    expect(
      clientId(uuidV1(), {
        type: types.AUTH_LOGOUT
      })
    ).toEqual(null);
  });

  it('Handle AUTH_LOGIN_SUCCESS', () => {
    expect(
      clientId(uuidV1(), {
        type: types.AUTH_LOGIN_SUCCESS
      })
    ).toMatch(/^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i);
  });
});

describe('auth reducer login', () => {

  it('should return the initial state', () => {
    expect(
      login(undefined, {})
    ).toEqual(asyncRequest)
  });

  it('Handle AUTH_LOGIN_REQUEST', () => {
    expect(
      login(asyncRequest, {
        type: types.AUTH_LOGIN_REQUEST,
      })
    ).toEqual({
      loading: true,
      success: false,
      error: false,
    });
  });

  it('Handle AUTH_LOGIN_SUCCESS', () => {
    expect(
      login(asyncRequest, {
        type: types.AUTH_LOGIN_SUCCESS,
      })
    ).toEqual({
      loading: false,
      success: true,
      error: false,
      errorMessages: {},
    });
  });

  it('Handle AUTH_LOGIN_ERROR', () => {
    expect(
      login(asyncRequest, {
        type: types.AUTH_LOGIN_ERROR,
        error: {
          msg: "Bad credential"
        }
      })
    ).toEqual({
      loading: false,
      success: false,
      error: true,
      errorMessages: { msg: "Bad credential" },
    });

  });

});

describe('auth reducer register', () => {

  it('should return the initial state', () => {
    expect(
      register(undefined, {})
    ).toEqual(asyncRequest)
  });

  it('Handle AUTH_REGISTER_REQUEST', () => {
    expect(
      register(asyncRequest, {
        type: types.AUTH_REGISTER_REQUEST,
      })
    ).toEqual({
      loading: true,
      success: false,
      error: false,
    });
  });


  it('Handle AUTH_LOGIN_SUCCESS', () => {
    expect(
      register(asyncRequest, {
        type: types.AUTH_LOGIN_SUCCESS,
      })
    ).toEqual({
      loading: false,
      success: true,
      error: false,
      errorMessages: {}
    });
  });

  it('Handle AUTH_REGISTER_ERROR', () => {
    expect(
      register(asyncRequest, {
        type: types.AUTH_REGISTER_ERROR,
        error: {
          msg: 'error when registering'
        }
      })
    ).toEqual({
      loading: false,
      success: false,
      error: true,
      errorMessages: { msg: 'error when registering' }
    });
  });
});


describe('auth reducer groups', () => {

  it('should return the initial state', () => {
    expect(
      groups(undefined, {})
    ).toEqual([])
  });

  it('Handle GROUP_LOAD_SUCCESS', () => {
    expect(
      groups([], {
        type: types.GROUP_LOAD_SUCCESS,
        groups: [
          { name: 'group1', is_personal: false },
          { name: 'jdoe', is_personal: true }
        ]
      })
    ).toEqual([
      GroupRecord({ name: 'group1', isPersonal: false }),
      GroupRecord({ name: 'jdoe', isPersonal: true })
    ])
  });

  it('Handle GROUP_CREATE_SUCCESS', () => {
    expect(
      groups([
        GroupRecord({ name: 'group1', isPersonal: false }),
        GroupRecord({ name: 'jdoe', isPersonal: true })
      ], {
        type: types.GROUP_CREATE_SUCCESS,
        group: { name: 'newGroup', is_personal: false },
      })
    ).toEqual([
      GroupRecord({ name: 'group1', isPersonal: false }),
      GroupRecord({ name: 'jdoe', isPersonal: true }),
      GroupRecord({ name: 'newGroup', isPersonal: false }),
    ])

  });

  it('Handle AUTH_LOGOUT', () => {
    expect(
      groups([
        GroupRecord({ name: 'group1', isPersonal: false }),
        GroupRecord({ name: 'jdoe', isPersonal: true })
      ], {
        type: types.AUTH_LOGOUT,
      })
    ).toEqual([])
  });
});

describe('auth reducer createGroup', () => {

  it('should return the initial state', () => {
    expect(
      createGroup(undefined, {})
    ).toEqual(asyncRequest)
  });

  it('Handle GROUP_CREATE_ASYNC', () => {
  });
  it('Handle GROUP_CREATE_SUCCESS', () => {
  });
  it('Handle GROUP_CREATE_ERROR', () => {
  });
});
