"""
Signals for notes app
"""
import logging

from django.contrib.auth import get_user_model
from django.contrib.auth.models import Group
from django.db.models.signals import post_save
from django.dispatch import receiver

from notes import constants
from notes.models import GroupProfile, UserProfile
from protocols.models import Protocol

User = get_user_model()
logger = logging.getLogger(__name__)

@receiver(post_save, sender=GroupProfile, dispatch_uid="group_profile_created_signal")
def group_profile_saved_callback(sender, instance, **kwargs):
    created = kwargs.pop('created')
    if instance and not instance.protocol:
        new_protocol = Protocol.objects.create_from_default(
            "Protocol group %s" % (instance.group.name or instance.group.ext_id ),
            "Protocol group %s" % (instance.group.name or instance.group.ext_id),
            instance.group.name,
            None
        )
        instance.protocol = "%s%s.%s" % (constants.PROTOCOL_URN_PREFIX,new_protocol.ext_id, new_protocol.revisions.last().version)
        instance.save()


@receiver(post_save, sender=Group, dispatch_uid="group_created_signal")
def group_saved_callback(sender, instance, **kwargs):
    created = kwargs.pop('created')
    if instance and created:
        profile = GroupProfile(group=instance)
        profile.save()


@receiver(post_save, sender=User, dispatch_uid="user_created_signal")
def user_saved_callback(sender, instance, **kwargs):
    created = kwargs.pop('created')
    if not instance:
        return
    base_group_name = instance.username + " group"
    group_name = base_group_name
    if created:
        # create personal group
        personal_group = None
        personal_group_profile = GroupProfile.objects.filter(owner=instance, owner_personal=instance).first()
        if personal_group_profile is None:
            # find a new name
            i = 1
            while Group.objects.filter(name=group_name).exists(): # should always ends...
                group_name = "%s %s" % (base_group_name, i)
                i += 1
            personal_group = Group.objects.create(name=group_name)
            personal_group.user_set.add(instance)
            personal_group_profile = personal_group.profile
            personal_group_profile.description = "%s personal group" % instance.username
            personal_group_profile.owner = instance
            personal_group_profile.owner_personal = instance

            personal_group_profile.save()
        else:
            personal_group = personal_group_profile.group

        UserProfile.objects.create(user=instance, default_group=personal_group)
    # else we do nothing, because we do not know if the group's name was changed or not
    # and we do not know the user older name
