"""
Tests the core api for sessions
"""
import logging
from uuid import uuid4

from django.contrib.auth import get_user_model
from django.urls import reverse
from django.utils import timezone
from rest_framework import status
from rest_framework.test import APITransactionTestCase
from django.contrib.auth.models import Group

logger = logging.getLogger(__name__)


class GroupApiTests(APITransactionTestCase):

    def setUp(self):
        User = get_user_model()
        self.user1 = User.objects.create_user(
            username='test_user1',
            email='test_user@email.com',
            password='top_secret'
        )
        self.user2 = User.objects.create_user(
            username='test_user2',
            email='test_user@email.com',
            password='top_secret'
        )

        self.group1 = Group(name='group1')
        self.group1.save()
        self.group1.user_set.add(self.user1)
        self.group1.profile.owner = self.user1
        self.group1.profile.description = "This is the group 1"
        self.group1.profile.save()

        self.group2 = Group(name='group2')
        self.group2.save()
        self.group2.profile.owner = self.user2
        self.group2.profile.description = "This is the group 2"
        self.group2.profile.save()


    def test_list_group_no_login(self):
        url = reverse('auth_group-list')
        response = self.client.get(url)
        logger.debug("LIST group response %r", response.data)

        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED)


    def test_list_group(self):
        url = reverse('auth_group-list')
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url)

        self.assertEqual(response.status_code, status.HTTP_200_OK)

        resp_json = response.json()
        self.assertIn('count', resp_json)
        self.assertIn('results', resp_json)
        self.assertEqual(4, resp_json['count'], "Must have 4 groups")
        self.assertEqual(4, len(resp_json['results']), "Must have 4 groups")

        for group_def in resp_json['results']:
            self.assertIn('name', group_def)
            self.assertIn('owner', group_def)
            self.assertIn('is_personal', group_def)
            self.assertIn('description', group_def)
            self.assertIn(group_def.get('owner'), ['test_user1', 'test_user2'])
            self.assertIn(group_def.get('description'), [
                'This is the group 1',
                'This is the group 2',
                'test_user1 personal group',
                'test_user2 personal group'])
            self.assertIn(group_def.get('name'), ['group1', 'group2', 'test_user1 group', 'test_user2 group'])

    def test_create_group(self):
        url = reverse('auth_group-list')
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.post(
            url,
            {'name':"group3", 'description': "this is group 3", 'users': ['test_user2']})

        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

        self.assertEqual('group3', response.data.get('name'))
        self.assertSetEqual(set(['test_user1', 'test_user2']), set(response.data.get('users')))

        group3 = Group.objects.get(name='group3')
        self.assertEqual('this is group 3', group3.profile.description)
        self.assertSetEqual(
            set(['test_user1', 'test_user2']),
            set(map(lambda u: u.username, group3.user_set.all()))
        )

    def test_create_group_no_user(self):
        url = reverse('auth_group-list')
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.post(
            url,
            {'name':"group3", 'description': "this is group 3"})

        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

        self.assertEqual('group3', response.data.get('name'))
        self.assertSetEqual(set(['test_user1']), set(response.data.get('users')))

        group3 = Group.objects.get(name='group3')
        self.assertEqual('this is group 3', group3.profile.description)
        self.assertSetEqual(
            set(['test_user1']),
            set(map(lambda u: u.username, group3.user_set.all()))
        )


    def test_detail_group(self):
        url = reverse('auth_group-detail', kwargs={'name': self.group1.name})
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.get(url)

        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual('group1', response.data.get('name'))
        self.assertEqual('This is the group 1', response.data.get('description'))
        self.assertSetEqual(set(['test_user1']), set(response.data.get('users')))

    def test_update_group(self):
        url = reverse('auth_group-detail', kwargs={'name': self.group1.name})
        self.client.login(username='test_user1', password='top_secret')
        response = self.client.put(
            url,
            {'name': 'group1', 'description': "this is group 1 changed", 'users': ['test_user2']})

        logger.debug("RESPONSE %r", response.data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual('group1', response.data.get('name'))
        self.assertEqual('this is group 1 changed', response.data.get('description'))
        self.assertSetEqual(set(['test_user1', 'test_user2']), set(response.data.get('users')))

        group1 = Group.objects.get(name='group1')
        self.assertEqual('this is group 1 changed', group1.profile.description)
        self.assertSetEqual(
            set(['test_user1', 'test_user2']),
            set(map(lambda u: u.username, group1.user_set.all()))
        )
