import PropTypes from 'prop-types';
import React, { Component } from 'react';
import { connect } from 'react-redux';
import { withRouter } from 'react-router';
import { bindActionCreators } from 'redux';
// import logo from './logo.svg';
import { Navbar, Nav, NavItem, NavDropdown, MenuItem, Modal, Button } from 'react-bootstrap';
import * as authActions from '../actions/authActions';
import { forceSync } from '../actions/networkActions';
import { groupSetCurrent } from '../actions/groupActions';
import { isAuthenticated, getCurrentUser, getOnline, getCurrentGroup, getGroups } from '../selectors/authSelectors';
import { isSynchronizing, isSynchronized } from '../selectors/syncSelectors';
import './Navbar.css';

const LoginNav = ({isAuthenticated, currentUser, history, authActions, onLogout}) => {

  const onClickSettings = (e) => {
    e.preventDefault();
    history.push('/settings');
  }

  const onClickLogin = (e) => {
    e.preventDefault();
    history.push('/login');
  }

  if (isAuthenticated) {
    return (
      <NavDropdown title={ currentUser.get('username') } id="user-dropdown">
        <MenuItem onClick={onClickSettings}>Settings</MenuItem>
        <MenuItem onClick={onLogout}>Logout</MenuItem>
      </NavDropdown>
    );
  }
  return (
    <NavItem onClick={onClickLogin} href="/login">Login</NavItem>
  );
}

const Online = ({ online }) => {
  return (
    <NavItem>
      <span className="material-icons" style={{ color: online ? '#2ECC71' : '#E74C3C' }}>signal_wifi_4_bar</span>
    </NavItem>
  )
}

const SyncButton = ({ onSyncClick, isSynchronizing, isSynchronized, id }) => {
  const classnames = "material-icons"
    + ((!isSynchronized)?" sync-button-not-synchronized":"")
    + ((isSynchronizing)?" sync-button-synchronizing":"");
  let title = "Synchronize";
  let clickCb = onSyncClick;
  if(isSynchronizing) {
    title = "Synchronizing...";
    clickCb = () => {};
  } else if (!isSynchronized) {
    title += ": not synchronized";
  }

  return (
    <NavItem title={title} onClick={clickCb} id={id || null}>
      <span className={classnames}>&#xE627;</span>
    </NavItem>
  )
}

const GroupStatus = ({currentGroup, groups, onSelect}) => {

  if(currentGroup) {
    const currentGroupName = currentGroup.get('name');
    return (
      <NavDropdown title={currentGroupName} id="group-dropdown" onSelect={onSelect}>
        { groups && groups.map((group, key) => {
            const groupName = group.get('name');
            const className = (groupName === currentGroupName)?'active':null;
            return <MenuItem className={className} key={key} eventKey={groupName}>{ groupName }</MenuItem>
          }
        )}
        <MenuItem key="-1" eventKey="__create_group__">Créer un groupe...</MenuItem>
      </NavDropdown>
    )
  } else {
    return null;
  }

}

class AppNavbar extends Component {

  state = {
    showModal: false
  }

  closeModal = () => {
    this.setState({ showModal: false });
  }

  onClickHome = (e) => {
    e.preventDefault();
    this.props.history.push('/');
  }

  isSynchronized = () => {
    return this.props.isSynchronized;
  }

  onClickLogout = (e) => {
    e.preventDefault();
    const isSynchronized = this.isSynchronized();
    if (isSynchronized) {
      this.logout();
    } else {
      this.setState({ showModal: true })
    }
  }

  confirmLogout = () => {
    const isSynchronized = this.isSynchronized();
    if (!isSynchronized) {
      this.props.authActions.resetAll();
    }
    this.logout();
    this.closeModal();
  }

  logout = () => {
    this.props.authActions.logout();
    this.props.history.push('/');
  }

  onClickSessions = (e) => {
    e.preventDefault();
    this.props.history.push('/sessions');
  }

  onSyncClick = (e) => {
    e.preventDefault();
    this.props.networkActions.forceSync();
  }

  onGroupSelect = (groupName) => {
    if(groupName === "__create_group__") {
      this.props.history.push('/create-group');
    } else {
      this.props.groupActions.groupSetCurrent(groupName);
    }
  }

  render() {
    return (
      <Navbar fluid inverse fixedTop>
        <Navbar.Header>
          <Navbar.Brand>
            <a onClick={this.onClickHome} href="/">IRI Notes</a>
          </Navbar.Brand>
          <Navbar.Toggle />
        </Navbar.Header>
        <Navbar.Collapse>
          <Nav>
            <NavItem onClick={this.onClickSessions} href="/sessions">Sessions</NavItem>
          </Nav>
          <Nav pullRight>
            <GroupStatus currentGroup={this.props.currentGroup} groups={this.props.groups} onSelect={this.onGroupSelect}/>
            <SyncButton id='sync-button' onSyncClick={this.onSyncClick} isSynchronizing={this.props.isSynchronizing} isSynchronized={this.props.isSynchronized} />
            <Online {...this.props} />
            <LoginNav {...this.props} onLogout={this.onClickLogout} />
          </Nav>
        </Navbar.Collapse>
        <Modal show={this.state.showModal} onHide={this.closeModal}>
          <Modal.Body>
            <p className="text-center">
              Some data is not synchronized with server.
              <br />
              If you continue, it will be lost.
            </p>
          </Modal.Body>
          <Modal.Footer>
            <Button bsStyle="primary" onClick={this.confirmLogout}>Confirm</Button>
            <Button onClick={this.closeModal}>Close</Button>
          </Modal.Footer>
        </Modal>
      </Navbar>
    );
  }
}

AppNavbar.propTypes = {
  isAuthenticated: PropTypes.bool.isRequired
};

function mapStateToProps(state, props) {
  return {
    isAuthenticated: isAuthenticated(state),
    currentUser: getCurrentUser(state),
    online: getOnline(state),
    isSynchronizing: isSynchronizing(state),
    isSynchronized: isSynchronized(state),
    currentGroup: getCurrentGroup(state),
    groups: getGroups(state)
  };
}

function mapDispatchToProps(dispatch) {
  return {
    authActions: bindActionCreators(authActions, dispatch),
    networkActions: bindActionCreators({ forceSync }, dispatch),
    groupActions: bindActionCreators({ groupSetCurrent }, dispatch)
  }
}

export default connect(mapStateToProps, mapDispatchToProps)(withRouter(AppNavbar));
