import os
import re
import sys
from distutils.command.install import INSTALL_SCHEMES
from distutils.command.install_data import install_data
from shutil import move
from tempfile import mkstemp

from setuptools import Command

try:
    from setuptools import setup
except ImportError:
    from distutils.core import setup


"""
See: https://dankeder.com/posts/adding-custom-commands-to-setup-py/
"""
class SetVersion(Command):
    """
    Set the version for the project
    """
    description = "Set the version for the project"
    # command_consumes_arguments = True
    user_options = [
        ('version=', 'v', 'version str')
    ]

    def initialize_options(self):
        """Set default values for options."""
        # Each user option must be listed here with their default value.
        self.version = None

    def finalize_options(self):
        """Post-process options."""
        if self.version is None:
            raise Exception("Parameter --version is missing")
        version_matches = re.match(r"(\d+).(\d+)(?:\.(\d+))?(?:(?:.(dev))|(?:(a|b|rc)(.*)))?", self.version)
        if version_matches is None:
            raise Exception("Parameter --version is not a version string (see )")
        version_list = version_matches.groups()
        version_tag = {'a': 'alpha', 'b': 'beta', 'rc': 'rc'}.get(version_list[4],'final')
        version_ext = int(version_list[5] or 0)
        if version_list[3] and version_list[3] == 'dev':
            version_tag = 'alpha'
            version_ext = 0

        self.version_tuple = (
            int(version_list[0]),
            int(version_list[1]),
            int(version_list[2] or 0),
            version_tag,
            version_ext,
        )

    def run(self):
        filepath = os.path.join(os.path.dirname(os.path.realpath(__file__)),"irinotes","__init__.py")
        print("Changing VERSION in %s to %r" % (filepath, self.version_tuple))
        fh, abs_path = mkstemp()
        with os.fdopen(fh,'w') as temp_file:
            with open(filepath, 'r') as initfiles:
                for l in initfiles:
                    if re.match(r"\s*VERSION\s*=\s*\(\d+\s*,\s*\d+\s*,\s*\d+,\s*[\"\'].+[\"\'],\s*.+\s*\)", l):
                        temp_file.write("VERSION = %r\n" % (self.version_tuple,))
                    else:
                        temp_file.write(l)
        os.remove(filepath)
        move(abs_path, filepath)



class osx_install_data(install_data):
    """
    On MacOS, the platform-specific lib dir is /System/Library/Framework/Python/.../
    which is wrong. Python 2.5 supplied with MacOS 10.5 has an Apple-specific fix
    for this in distutils.command.install_data#306. It fixes install_lib but not
    install_data, which is why we roll our own install_data class.
    """

    def finalize_options(self):
        """
        By the time finalize_options is called, install.install_lib is set to the
        fixed directory, so we set the installdir to install_lib. The
        install_data class uses ('install_data', 'install_dir') instead.
        """
        self.set_undefined_options('install', ('install_lib', 'install_dir'))
        install_data.finalize_options(self)

def fullsplit(path, result=None):
    """
    Split a pathname into components (the opposite of os.path.join) in a
    platform-neutral way.
    """
    if result is None:
        result = []
    head, tail = os.path.split(path)
    if head == '':
        return [tail] + result
    if head == path:
        return result
    return fullsplit(head, [tail] + result)


def launch_setup(setup_script_name, setup_script_args):
    """
    Start setup
    """
    if sys.platform == "darwin":
        cmdclasses = {'install_data': osx_install_data}
    else:
        cmdclasses = {'install_data': install_data}


    root_dir = os.path.dirname(__file__)
    if root_dir != '':
        os.chdir(root_dir)
    source_dirs = ['irinotes', 'notes', 'protocols']

    version_variables = {}
    try:
        with open(os.path.join(source_dirs[0], "__init__.py")) as f:
            code = compile(f.read(), "__init__.py", 'exec')
            exec(code, version_variables)
    except:
        pass

    version = version_variables['__version__']

    packages, data_files = [], []

    for source_dir in source_dirs:
        for dirpath, dirnames, filenames in os.walk(source_dir):
            # Ignore dirnames that start with '.'
            for i, dirname in enumerate(dirnames):
                if dirname.startswith('.') or dirname.startswith('__pycache__'): del dirnames[i]
            if '__init__.py' in filenames:
                packages.append('.'.join(fullsplit(dirpath)))
            elif filenames:
                data_files.append([dirpath, [os.path.join(dirpath, f) for f in filenames]])


    # Tell distutils to put the data_files in platform-specific installation
    # locations. See here for an explanation:
    # http://groups.google.com/group/comp.lang.python/browse_thread/thread/35ec7b2fed36eaec/2105ee4d9e8042cb
    for scheme in INSTALL_SCHEMES.values():
        scheme['data'] = scheme['purelib']

    # Small hack for working with bdist_wininst.
    # See http://mail.python.org/pipermail/distutils-sig/2004-August/004134.html
    if len(sys.argv) > 1 and sys.argv[1] == 'bdist_wininst':
        for file_info in data_files:
            file_info[0] = '\\PURELIB\\%s' % file_info[0]

    #write MANIFEST.in

    with open("MANIFEST.in", "w") as m:
        m.write("include CHANGES\n")
        m.write("include LICENSE\n")
        m.write("include README\n")
        m.write("include MANIFEST.in\n")
        m.write("include requirements/base.txt\n")
        m.write("include requirements/dev.txt\n")
        m.write("include requirements/prod.txt\n")
        for entry in data_files:
            file_list = entry[1]
            for filename in file_list:
                m.write("include %s\n" % (filename))

    long_description = ''
    with open('README', 'r') as f:
        long_description = f.read()

    cmdclasses = {**cmdclasses, **{ 'set_version': SetVersion }}

    setup(
        script_name=setup_script_name,
        script_args=setup_script_args,
        name='irinotes',
        version=version,
        author='IRI',
        author_email='contact@iri.centrepompidou.fr',
        packages=packages,
        data_files=data_files,
        cmdclass=cmdclasses,
        scripts=[],
        url='http://www.iri.centrepompidou.fr/dev/hg/irinotes',
        license='CECILL-C',
        description='projet Irinotes',
        long_description=long_description,
        classifiers=[
            'Development Status :: 4 - Beta',
            'Environment :: Web Environment',
            'Framework :: Django',
            'Intended Audience :: Developers',
            'License :: Ceccil-B',
            'Operating System :: OS Independent',
            'Programming Language :: Python',
            'Topic :: Utilities'
        ],
        install_requires=[
            "Django >= 1.11",
            "python-decouple",
            "Unipath",
            "dj-database-url",
            "six",
            "django-auditlog @ https://github.com/IRI-Research/django-auditlog/tarball/master#egg=django-auditlog-0.4.5+IRI",
            "django-extensions",
            "djangorestframework >= 3.6",
            "django-rest-auth[with_social]",
            "djangorestframework-jwt",
            "django-guardian >= 1.4",
            "django-colorful",
            "django-cors-headers",
            "django-concurrency",
            "django-filter",
            "drf-nested-routers",
            "markdown"
        ],
        # dependency_links=[
        #     "https://github.com/IRI-Research/django-auditlog/tarball/master#egg=django-auditlog-0.4.5dev"
        # ]
    )


if __name__ == "__main__":

    script_name = os.path.basename(sys.argv[0])
    script_args = sys.argv[1:]

    launch_setup(script_name, script_args)
