from django.db import models, transaction
from django.conf import settings
from django.contrib.auth.models import User
from django.contrib.contenttypes.fields import GenericForeignKey
from django.contrib.contenttypes.models import ContentType
import uuid, json


class Tag(models.Model):
	label = models.SlugField(blank=True, null=True)
	link = models.URLField(unique=True)
	description = models.CharField(max_length=255, blank=True, null=True)
	collection = models.ForeignKey('Collection', blank=True, null=True)
	
	def is_internal(self):
		return link.startswith(settings.BASE_URL)

class TaggingInfo(models.Model):
	revision = models.ForeignKey('AnnotationRevision', on_delete=models.CASCADE)
	tag = models.ForeignKey('Tag', on_delete=models.CASCADE)
	accuracy = models.IntegerField()
	relevancy = models.IntegerField()
	

# fonds Ingres - Musee de la Poste 
class Collection(models.Model):
	name =	models.CharField(max_length=50, unique=True)
	description = models.CharField(max_length=255)

	def __str__(self):
		return self.name


class ItemMetadata(models.Model):
	item = models.OneToOneField('Item', related_name='metadatas')
	joconde_ref = models.CharField(max_length=20, null=False, blank=False, unique=True)
	domain = models.CharField(max_length=255)
	title = models.CharField(max_length=255)
	description = models.CharField(max_length=255)

class Item(models.Model):
	collection = models.ForeignKey(Collection, related_name="items")

class ImageStats(models.Model):
	image = models.OneToOneField('Image', related_name='stats', blank=False, null=False)
	views_count = models.IntegerField(blank=True, null=True, default=0)
	annotations_count = models.IntegerField(blank=True, null=True, default=0)
	submitted_revisions_count = models.IntegerField(blank=True, null=True, default=0)
	comments_count = models.IntegerField(blank=True, null=True, default=0)
	folders_inclusion_count = models.IntegerField(blank=True, null=True, default=0)
	tag_count = models.IntegerField(blank=True, null=True, default=0)

class Image(models.Model):
	image_guid = models.UUIDField(default=uuid.uuid4, editable=False)
	name = models.CharField(max_length=200)
	media = models.ImageField(upload_to='uploads/', height_field='height', width_field='width')
	item = models.ForeignKey('Item', related_name='images', null=True, blank=True)
	height = models.IntegerField(null=False, blank=False)
	width = models.IntegerField(null=False, blank=False)
	created = models.DateTimeField(auto_now_add=True, null=True)
	
	def __str__(self):
		return self.name

# # Folders
# class Folder(models.Model):
# 	label = models.CharField(max_length=255)
# 	owner = models.ForeignKey(User)
# 	images = models.ManyToManyField(Image)
# 
# 	def __str__(self):
# 		return label

class AnnotationManager(models.Manager):
	
	# Call Annotation.objects.create_annotation to initialize a new Annotation with its associated AnnotationStats and initial AnnotationRevision
	@transaction.atomic
	def create_annotation(self, author, image, title='', description='', fragment='', tags_json='[]'):
		# Create annotation object
		new_annotation = Annotation(
			image=image, 
			author=author
		)
		new_annotation.save()
		
		# Create initial revision
		initial_revision = AnnotationRevision(
			annotation=new_annotation, 
			author=author,
			title=title,
			description=description,
			fragment=fragment,
			state=AnnotationRevision.ACCEPTED
		)
		initial_revision.save()
		initial_revision.set_tags(tags_json)
		
		new_annotation.current_revision = initial_revision
		new_annotation.save()
		
		# Create stats object
		new_annotation_stats = AnnotationStats(annotation=new_annotation)
		new_annotation_stats.save()
		new_annotation.stats = new_annotation_stats
		new_annotation.save()
		return new_annotation

class AnnotationStats(models.Model):
	annotation = models.OneToOneField('Annotation', related_name='stats', blank=False, null=False)
	submitted_revisions_count = models.IntegerField(blank=True, null=True, default=1)
	accepted_revisions_count = models.IntegerField(blank=True, null=True, default=1)
	contributors_count = models.IntegerField(blank=True, null=True, default=1)
	views_count = models.IntegerField(blank=True, null=True, default=0)
	comments_count = models.IntegerField(blank=True, null=True, default=0)
	tag_count = models.IntegerField(blank=True, null=True, default=0)
	
	def contributors(self):
		# returns users that submitted an accepted revision
		return


class Annotation(models.Model):
	annotation_guid = models.UUIDField(default=uuid.uuid4, editable=False)
	image = models.ForeignKey('Image', related_name='annotations', on_delete=models.CASCADE)
	source_revision = models.ForeignKey('AnnotationRevision', related_name='source_related_annotation', blank=True, null=True)
	current_revision = models.OneToOneField('AnnotationRevision', related_name='current_for_annotation', blank=True, null=True)
	author = models.ForeignKey(User, null=True)
	created = models.DateTimeField(auto_now_add=True, null=True)
	
	objects = AnnotationManager()
	
	def update_stats(self):
		pass
	
	# Call to create a new revision, possibly from a merge
	@transaction.atomic
	def make_new_revision(self, author, title, description, fragment, tags):
		if author == self.author:
			# We're creating an automatically accepted revision
			new_revision_state = AnnotationRevision.ACCEPTED
		else:
			# Revision will require validation
			new_revision_state = AnnotationRevision.AWAITING
		new_revision = AnnotationRevision(
			annotation = self,
			parent_revision=self.current_revision,
			title=title,
			description=description,
			author=author,
			fragment=fragment,
			state=new_revision_state
		)
		new_revision.save()
		if new_revision.state == AnnotationRevision.ACCEPTED:
			self.current_revision = new_revision
			self.save()
		return new_revision
	
	# Call when we're validating an awaiting revision whose parent is the current revision AS IT WAS CREATED
	@transaction.atomic
	def validate_existing_revision(self, revision_to_validate):
		if revision_to_validate.parent_revision == current_revision:
			self.current_revision = revision_to_validate
			revision_to_validate.state = AnnotationRevision.ACCEPTED
			revision_to_validate.save()
			self.save()
			
	# Call when we're validating an awaiting revision whose parent isn't the current revision OR IF IT WAS CHANGED BY THE ANNOTATION AUTHOR
	@transaction.atomic
	def merge_existing_revision(self, title, description, fragment, tags, revision_to_merge):
		merged_revision = self.make_new_revision(author=self.author, title=title, description=description, fragment=fragment, tags=tags)
		merged_revision.merge_parent_revision = revision_to_merge
		merged_revision.save()
		self.current_revision=merged_revision
		self.save()

class AnnotationRevision(models.Model):
	
	AWAITING = 0
	ACCEPTED = 1
	REJECTED = 2
	
	REVISION_STATES = (
		(AWAITING, 'awaiting'),
		(ACCEPTED, 'accepted'),
		(REJECTED, 'rejected')
	)
	
	
	revision_guid = models.UUIDField(default=uuid.uuid4)
	annotation = models.ForeignKey('Annotation', related_name='revisions', null=False, blank=False)
	parent_revision = models.ForeignKey('AnnotationRevision', related_name='reverse_parent_revision', blank=True, null=True)
	merge_parent_revision = models.ForeignKey('AnnotationRevision', related_name='reverse_merge_parent_revision', blank=True, null=True)
	author = models.ForeignKey(User, null=True)
	title = models.CharField(max_length=255)
	description = models.TextField(null=True)
	fragment = models.TextField()
	tags = models.ManyToManyField('Tag', through='TaggingInfo', through_fields=('revision', 'tag'), blank=True, null=True)
	state = models.IntegerField(choices=REVISION_STATES, default=AWAITING)
	created = models.DateTimeField(auto_now_add=True, null=True)

	def set_tags(self, tags_json_string):
		try:
			tags_dict = json.loads(tags_json_string)
		except ValueError:
			pass
		for tag_data in tags_dict:
			tag_string = tag_data.get("tag_input")
			tag_accuracy = tag_data.get("accuracy")
			tag_relevancy = tag_data.get("relevancy")
			
			if tag_string.startswith("http://"): #check if url
				if Tag.objects.exists(link=tag_string): #check if tag already exists
					tag_obj = Tag.objects.get(link=tag_string)
				else:
					tag_obj = Tag.objects.create(
						link = settings.BASE_URL+tag_string,
					)
			else:
				tag_obj = Tag.objects.create(
					label = tag_string,
					description = "",
					link = settings.BASE_URL+tag_string,
					collection = self.parent_annotation.image.item.collection
				)
			tag_info = TaggingInfo.objects.create(
				tag=tag_obj, 
				revision=self,
				accuracy = tag_accuracy,
				relevancy = tag_relevancy
			)
		
		
	
#class MetaCategory(models.Model):
#	collection = models.ForeignKey(Collection)
#	label = models.CharField(max_length=200)
#
#	def __str__(self):
#		return self.label
#
#	class Meta:
#		verbose_name_plural = 'Metacategories'
#
# 
# class Comment(models.Model):
# 	author =  models.ForeignKey(User)
# 	created = models.DateTimeField(blank=False, null=False, auto_now_add=True)
# 	annotation_revision = models.ForeignKey(AnnotationRevision, blank=True, null=True)
# 	target = models.ForeignKey('Comment', blank=True, null=True)
# 	content = models.TextField(blank=True)
# 	metacategories = models.ManyToManyField(MetaCategory) 
# 
# 
# class CommentAttachement(models.Model):
# 	
# 	LINK = 0
# 	IMAGE = 1
# 	PDF = 2
# 	COMMENT_CHOICES = (
# 		(LINK, 'link'),
# 		(IMAGE, 'image'),
# 		(PDF, 'pdf')
# 	)
# 	comment = models.ForeignKey(Comment, on_delete=models.CASCADE, related_name='attachments')
# 	main_annotation = models.ForeignKey(Annotation)
# 	attachment_type = models.IntegerField(choices=COMMENT_CHOICES, default=0)
# 	created_date = models.DateTimeField(auto_now_add=True)
# 	data = models.TextField(blank=False)
# 
# 
# # Activity & Notification
# 
# class Activity(models.Model):
# 	
# 	NEW_COMMENT = 0
# 	NEW_REVISION = 1
# 	NEW_COMMENT_ON_REVISION = 2
# 	NEW_EXPERT_CALL = 3
# 	NEW_EXPERT_ANSWER = 4
# 	NEW_REFERENCE = 5
# 	
# 	ACTIVITY_VERBS = (
# 		(NEW_COMMENT, 'New comment'),
# 		(NEW_REVISION, 'New revision'),
# 		(NEW_COMMENT_ON_REVISION, 'New comment on a revision'),
# 		(NEW_EXPERT_CALL, 'New expert call'),
# 		(NEW_EXPERT_ANSWER, 'New expert answer'),
# 		(NEW_REFERENCE, 'New reference'),
# 	) 
# 
# 	verb = models.IntegerField(choices=ACTIVITY_VERBS)
# 	actor = models.ForeignKey(User)
# 
# 	target_content_type = models.ForeignKey(ContentType, on_delete=models.CASCADE)
# 	target_object_id = models.PositiveIntegerField()
# 	target = GenericForeignKey('target_content_type', 'target_object_id')
# 
# 	action_content_type = models.ForeignKey(ContentType, related_name='activity_action', on_delete=models.CASCADE, null=True, blank=True)
# 	action_object_id = models.PositiveIntegerField(null=True, blank=True)
# 	action_content = GenericForeignKey('action_content_type', 'action_object_id') 
# 	
# 	created_date = models.DateTimeField(auto_now_add=True)
# 
# 	def __str__(self):
# 		return '%s:%s' % (author.name, verbe)
# 
# 
# class Notification(models.Model):
# 
# 	UNREAD = 0
# 	READ = 1
# 	DELETED = 2
# 
# 	STATUS = (
# 		(UNREAD, 'Unread'),
# 		(READ, 'Read'),
# 		(DELETED, 'Deleted')
# 	)
# 
# 	activity = models.ForeignKey(Activity)
# 	user = models.ForeignKey(User)
# 	status = models.IntegerField(choices=STATUS, default=UNREAD)
# 	created_date = models.DateTimeField(auto_now_add=True)

